import collections
try:
    import configparser
except ImportError:
    import ConfigParser as configparser

from sqlalchemy.orm import sessionmaker
from sqlalchemy.orm import scoped_session
from sqlalchemy import MetaData, create_engine
import sqlalchemy

from libtng import const
import libtng.environ

from connection import *
from utils import router
from schema import Relation
from renderdb import render


__all__ = [
    'ConnectionOptions',
    'connections',
    'default_connection',
    'ConnectionDoesNotExist',
    'router',
    'Relation',
    'get_session',
    'get_relation',
    'execute_script'
]


meta = MetaData()


def get_session(alias=None, *args, **kwargs):
    """
    Instantiate a new database session (cursor).

    :param alias:
        specifies the database connection to get a session
        on.
    :returns:
        a :class:`sqlalchemy.orm.session.Session` instance.
    """
    alias = alias or libtng.environ.getenv('DEFAULT_DB_ALIAS',
        const.DEFAULT_DB_ALIAS)
    session = connections.get_session(alias, scoped=True, *args, **kwargs)
    setattr(session, 'using', alias)
    return session


def get_relation(alias, relname, schema='public'):
    """
    Autoload a relation by querying the database catalog.

    :param alias:
        specifies the database connection to use.
    :param relname:
        the name of the relation.
    :param schema:
        the schema holding the relation. Default is ``'public'``.
    :returns:
        a :class:`libtng.db.schema.Relation` instance.
    """
    engine = connections[alias]
    return Relation(relname, meta, schema=schema, autoload=True,
        autoload_with=engine)


def construct_row_class(selectable):
    """
    Given a selectable, construct a :class:`namedtuple` from
    the selectables' columns. It is assumed that column names
    are valid Python identifiers.

    :param selectable:
        a selectable object.
    :returns:
        a namedtuple wrapping the column names.
    """
    if isinstance(selectable, sqlalchemy.Table):
        columns = [x.name for x in selectable.columns]
        classname = "{0}".format(relation.name.title())
    elif isinstance(selectable, sqlalchemy.orm.query.Query):
        classname = 'ResultSetTuple'
        columns = [x['name'] for x in selectable.column_descriptions]
    else:
        raise NotImplementedError
    cls = collections.namedtuple(classname, columns)
    return cls


def execute_script(alias, src, commit=True, is_ddl=False, is_template=False, **kwargs):
    """
    Execute the SQL statements defined in file located at `src`.

    :param alias:
        the database connection to use.
    :param src:
        a string pointing to a file on the local filesystem.
    :param is_ddl:
        a boolean indicating if the script containts DDL statements.
    :param is_template:
        indicates if the script is a template. If ``is_template=True``,
        the file will be rendered using :mod:`jinja2`.
    :returns:
        the executed SQL.
    """
    sql = open(src).read().replace('%','%%')
    if is_template:
        t = jinja2.Template(sql)
        sql = t.render(kwargs)
    ddl = sqlalchemy.schema.DDL(sql)
    if commit:
        session = connections.get_session(alias)
        session.begin()
        try:
            session.execute(ddl)
            session.commit()
        except Exception:
            session.rollback()
            raise
    return ddl


def parse_configuration(*filepaths):
    """
    Parses a database configuration file (as specified per documentation)
    and returns a dictionary containing :class:`sqlalchemy.Engine`
    instances.

    Args:
        filepaths: positional arguments specifying the filepath to
            database configuration files.

    Returns:
        dict: mapping database aliases to engines.
    """
    config = configparser.ConfigParser()
    config.read(filepaths)
    connections = collections.defaultdict(lambda: {
        'read': collections.defaultdict(list),
        'write': None
    })
    for section in config.sections():
        if not section.startswith('database:'):
            continue
        parts = section.split(':')
        if len(parts) == 2: # No type specified, default is write.
            alias, connection_type = parts[1], 'write'
        else:
            alias, connection_type = parts[1:]
        engine = create_engine(config.get(section, 'dsn'))\
            .execution_options(autocommit=True)
        if connection_type == 'write':
            connections[alias][connection_type] = {
                'engine': engine,
                'session_class': scoped_session(sessionmaker(bind=engine))
            }
        else:
            connections[alias][connection_type].append({
                'engine': engine,
                'session_class': scoped_session(sessionmaker(bind=engine))
            })
    return connections


class Connections(object):

    @classmethod
    def fromfiles(cls, *args):
        return cls(parse_configuration(*args))

    def __init__(self, connections):
        self._connections = connections

    def get_session(self, alias):
        return self._connections[alias]['write']['session_class']()