import abc

from libtng.provisioning.inventory import Inventory
from libtng.provisioning.operation import ProvisioningOperation


class Provisionable(object):
    """
    Base class for provisionable objects, used to invoke a
    provisioning operation.
    """

    @abc.abstractmethod
    def get_provisioner_hosts(self):
        """
        Return a list of hosts to operate on. The members of the
        list must be either :class:`Host` instances, or three-tuples
        composed of the hostname, a list containing the groups, and
        a :class:`dict` holding the host-specific variables.
        """
        raise NOTIMPLEMENTED_SUBCLASS

    def get_provisioner_module(self):
        """
        Get a module and it's arguments that the :class:`Provisioner`
        should invoke. Must return a tuple containing a string and
        a :class:`dict`, indicating the module name and the module
        arguments.

        When the object does not call an Ansible module, it
        **must** raise a :exc:`NotImplementedError`.
        """
        raise NOTIMPLEMENTED_SUBCLASS

    def get_provisioner_playbook(self):
        """
        Get the location of a playbook that the :class:`Provisioner`
        should run. Must return a tuple containing a string and
        a :class:`dict`, specifying the playbook filepath and
        additional context variables.

        When the object does not run an Ansible playbook, it
        **must** raise a :exc:`NotImplementedError`.
        """
        raise NOTIMPLEMENTED_SUBCLASS

    def get_provisioner_inventory(self):
        """
        Get the :class:`Inventory` used to determine which
        host(s) are to be provisioned.
        """
        inventory = Inventory()
        for host in self.get_provisioner_hosts():
            inventory.append(host)
        return inventory

    def get_provisioner_operation(self):
        """
        Return a :class:`ProvisioningOperation` instance
        indicating the type of provisioning that should
        occurr (module or playbook).
        """
        playbook_name = extra_params = module_name = module_args = None
        try:
            module_name, module_args = self.get_provisioner_module()
        except NotImplementedError:
            try:
                playbook_name, extra_params = self.get_provisioner_playbook()
            except NotImplementedError:
                pass
        if not any([playbook_name, module_name]):
            raise NotImplementedError("Provisionable must implement e"
                "ither get_provisioning_playbook() or get_provisioning_module()")
        return ProvisioningOperation(playbook_name=playbook_name,
            extra_params=extra_params, module_name=module_name,
            module_args=module_args)
