import re

from django.middleware.csrf import rotate_token
from django.core.handlers.wsgi import WSGIRequest
from werkzeug.datastructures import EnvironHeaders
from werkzeug.urls import url_decode
from werkzeug._compat import wsgi_get_bytes
from werkzeug.datastructures import ImmutableMultiDict

from libtng.functional import cached_property

# Mimic Werkzeug API for Django
# Imitates werkzeug request
class WSGIRequest(WSGIRequest):

    #: the class to use for `args` and `form`.  The default is an
    #: :class:`~werkzeug.datastructures.ImmutableMultiDict` which supports
    #: multiple values per key.  alternatively it makes sense to use an
    #: :class:`~werkzeug.datastructures.ImmutableOrderedMultiDict` which
    #: preserves order or a :class:`~werkzeug.datastructures.ImmutableDict`
    #: which is the fastest but only remembers the last key.  It is also
    #: possible to use mutable structures, but this is not recommended.
    #:
    #: .. versionadded:: 0.6
    parameter_storage_class = ImmutableMultiDict

    #: the charset for the request, defaults to utf-8
    charset = 'utf-8'

    #: the error handling procedure for errors, defaults to 'replace'
    encoding_errors = 'replace'

    header_pattern = re.compile('(\w+)[:=] ?"?([\w\s\d\.\-\_\:]+)"?')

    #: The names to use when the HTTP ``Basic`` authentication scheme
    #: is used.
    basic_auth_names = ['client_id', 'client_secret']

    @property
    def authorization(self):
        """Parses the Authorization header and returns a
        :class:`~libtng.http.authorization.Authorization`
        instance, or ``None`` if no header was present.

        May raise a :exc:`ValueError` if the contents
        of the ``Authorization`` header could not be
        parsed.
        """
        scheme, params = None, {}
        try:
            scheme, raw_params = self.headers['Authorization']\
                .split(' ', 1)
            if scheme == 'Basic':
                params = dict(zip(self.basic_auth_names,
                    raw_params.decode('base64').split(':')))
            else:
                params = self.header_pattern.findall(raw_params)
                if not params:
                    raise ValueError("Invalid header contents: {0}"\
                        .format(raw_params))
            return Authorization(scheme, collections.OrderedDict(params))
        except KeyError:
            return None

    @property
    def url_charset(self):
        """The charset that is assumed for URLs.  Defaults to the value
        of :attr:`charset`.

        .. versionadded:: 0.6
        """
        return self.encoding

    @cached_property
    def headers(self):
        """The headers from the WSGI environ as immutable
        :class:`~werkzeug.datastructures.EnvironHeaders`.
        """
        return EnvironHeaders(self.environ)

    @cached_property
    def form(self):
        return self.POST

    @cached_property
    def args(self):
        """The parsed URL parameters.  By default an
        :class:`~werkzeug.datastructures.ImmutableMultiDict`
        is returned from this function.  This can be changed by setting
        :attr:`parameter_storage_class` to a different type.  This might
        be necessary if the order of the form data is important.
        """
        return url_decode(wsgi_get_bytes(self.environ.get('QUERY_STRING', '')),
                          self.url_charset, errors=self.encoding_errors,
                          cls=self.parameter_storage_class)


    @cached_property
    def origin(self):
        """The IP address from which the incoming request originates."""
        return self.headers['Origin']

    def json(self):
        return json.loads(self.body)

    @property
    def files(self):
        return self.FILES

    def sign_in(self, session, user):
        """
        Persist a user id and a backend in the session. Note that
        data set during the anonymous session is retained when
        the user logs in.

        Args:
            session (django.contrib.session.backends.base.SessionBase):
                the session of the current application user.
            user (tngems.sousou.domain.User): the domain object
                representing the current user.

        Returns:
            None
        """
        if hasattr(self, 'user'):
            self.user = user
        rotate_token(self)