from libtng.datastructures import OrderedSet
from libtng.provisioning.provisionable import Provisionable
from libtng.provisioning.inventory import Inventory
from libtng.provisioning.utils import args_to_string


class ProvisionableCollection(Provisionable):
    """
    Represents a collection of :class:`Provisionable` instances.
    Will override any module or playbook options set on the
    provisionables.
    """

    @classmethod
    def as_playbook(cls, provisionables, playbook_name, extra_params):
        """
        Return a new :class:`ProvisionableCollection` configured to
        run an Ansible playbook.

        :param playbook_name:
            a string specifying the Ansible playbook to run.
        :param extra_params:
            a :class:`dict` containing extra parameters for the playbooks'
            template context.
        :returns:
            :class:`ProvisionableCollection`
        """
        return cls(provisionables, None, None, playbook_name, extra_params)

    @classmethod
    def as_module(cls, provisionables, module_name, **module_args):
        """
        Return a new :class:`ProvisionableCollection` configured to
        run an Ansible module.

        :param module_name:
            specifies the Ansible module to run.
        :param module_args:
            a dictionary containing the arguments to provide to the module;
            must be primitives and not sequences or mappings.
        :returns:
            :class:`ProvisionableCollection`
        """
        return cls(provisionables, module_name, module_args, None, None)

    def __init__(self, provisionables, module_name, module_args, playbook_name, extra_params):
        """
        Initializes a new :class:`ProvisionableCollection` instance.

        :param provisionables:
            a list of :class:`Provisionable` objects.
        :param module_name:
            specifies the Ansible module to run.
        :param module_args:
            a dictionary containing the arguments to provide to the module;
            must be primitives and not sequences or mappings.
        :param playbook_name:
            a string specifying the Ansible playbook to run.
        :param extra_params:
            a :class:`dict` containing extra parameters for the playbooks'
            template context.

        .. admonition::

            You must specify either ``(module_name, module_args)`` or
            ``(playbook_name, extra_params)``.
        """
        if not (bool(playbook_name) ^ bool(module_name)):
            raise ProgrammingError(
                "Provisionable must specify either a module or a playbook.")
        self._provisionables = OrderedSet(provisionables)
        self._module_name = module_name
        self._module_args = module_args or {}
        self._playbook_name = playbook_name
        self._extra_params = extra_params or {}

    def get_provisioner_inventory(self):
        # Aggregate the inventories of all provisionables into
        # a single object.
        inv = Inventory()
        for p in self._provisionables:
            inv &= p.get_provisioner_inventory()
        return inv

    def get_provisioner_module(self):
        if not self._module_name:
            raise NotImplementedError
        return self._module_name, args_to_string(**self._module_args)

    def get_provisioner_playbook(self):
        if not self._playbook_name:
            raise NotImplementedError
        return self._playbook_name, self._extra_params

    def get_provisioner_hosts(self):
        # get_provisioner_hosts() is a noop on collection.
        raise NotImplementedError(
            "Method get_provisioner_hosts() is not implemented on Prov"
            "isionableCollection")

    def __iter__(self):
        return iter(self._provisionables)

    def __len__(self):
        return len(self._provisionables)