import collections

from gateway import DuplicateCommandError


__all__ = ['Transaction']


class Transaction(collections.Sequence):
    """
    Represents a  collection of :class:`~libtng.cqrs.Command`
    instances composing a single unit of work.
    """

    def __init__(self, gateway, ignore_unique=False):
        """
        Initialize a new :class:`Transaction`.

        :param gateway:
            the gateway responsible for dispatching the commands in the
            :class:`Transaction`.
        :param ignore_unique:
            a bool indicating if unique violations on commands should fail
            silent.
        """
        self.__commands = []
        self.__gateway = gateway
        self.__ignore_unique = ignore_unique

    def append(self, command, fail_silent=False):
        """
        Appends a :class:`~libtng.cqrs.Command` to the transaction.
        """
        if command.is_unique() and command in self.__commands:
            if not self.__ignore_unique and not fail_silent:
                raise DuplicateCommandError(
                    "Command already in transaction: {0}".format(command))
            return False
        self.__commands.append(command)
        return True

    def commit(self):
        """
        Commits all commands and returns the results.
        """
        return [self.__gateway.dispatch(x) for x in self.__commands]

    def __getitem__(self, index):
        return self.__commands[index]

    def __len__(self):
        return len(self.__commands)