import copy

from libtng.exceptions import ProgrammingError


class HttpException(Exception):
    """Base class for exceptions that should result in a
    certain HTTP response.
    """
    status_code = None
    default_params = {}
    default_mimetype = "text/html"
    template_name = None

    def __init__(self, params=None, mimetype=None, headers=None, response_factory=None,
        hint=None, detail=None):
        """
        Initialize a new :class:`HttpException` instance.
        """
        if self.status_code is None:
            raise ProgrammingError("HttpException subclasses must define a `status_code` attribute.")
        self.params = copy.deepcopy(self.default_params)
        self.params.update(params or {})
        self.headers = headers
        self.mimetype = mimetype or self.default_mimetype
        self.response_factory = response_factory
        self.hint = hint
        self.detail = detail

    def render_to_response(self, request):
        """
        Get the content for the HTTP response to the client.

        Args:
            request: the incoming WSGI request object.

        Returns:
            HttpResponse: returned by the response factory provided
                to the constructor.
            None: no response factory was provided to the constructor.
        """
        if self.response_factory is not None:
            return self.response_factory(self.get_response_content(request),
                status_code=self.status_code, mimetype=self.mimetype)

    def get_params(self):
        """
        Returns the parameters specified to the exception constructor.
        """
        params = copy.deepcopy(self.params)
        params['status_code'] = self.status_code
        if self.hint:
            params['hint'] = self.get_hint()
        if self.detail:
            params['detail'] = self.detail
        return params

    def get_response_content(self, request):
        """
        Return the content of an outgoing response.
        """
        if self.mimetype == 'application/json':
            content = self.get_params()
        else:
            raise NotImplementedError
        return content

    def get_hint(self):
        return self.hint