import collections

from libtng.exceptions import ImproperlyConfigured
from libtng.cqrs.provider import registry
from libtng.cqrs.const import FAILED
from libtng.cqrs.const import FINISHED

__all__ = ['Runner']


CommandExecutionResult = collections.namedtuple('CommandExecutionResult',
    ['retval','state'])


class Runner(object):
    """
    Responsible for running commands. The default implementation
    only call the :meth:`libtng.cqrs.CommandHandler.handle` method
    on a command handler; override the :meth:`Runner.run` method
    for more fine-grained behavior, such as dependency injection,
    security checks, transaction management, logging, etc.
    """

    def __init__(self, handlers=None):
        """
        Initialize a new :class:`Runner` instance.

        Args:
            handlers (libtng.cqrs.HandlersProvider): the provider of command
                handlers.
        """
        self._handlers = handlers or registry

    def run(self, gateway, command):
        """
        Run the specified `command`.

        Args:
            gateway (libtng.cqrs.GatewayInterface): a concrete gateway
                implementation.
            command (libtng.cqrs.Command): a concret command implementation.

        Returns:
            libtng.cqrs.CommandExecutionResult
        """
        handler = self._handlers.get_handler(gateway, self, command)
        try:
            retval = handler.run_command(command, handler.handle)
            result = CommandExecutionResult(retval, FINISHED)
        except Exception as e:
            state = getattr(e, 'state', FAILED)
            result = CommandExecutionResult(retval, state)
        command.metadata.set_state(result.state)
        return result