"""
Various datastructures.
"""
import collections
import copy


__all__ = ['OrderedSet','ImmutableDict']


class ImmutableDict(collections.Mapping):

    def __init__(self, somedict):
        self._dict = dict(somedict)
        self._hash = None

    def __getitem__(self, key):
        return self._dict[key]

    def __len__(self):
        return len(self._dict)

    def __iter__(self):
        return iter(self._dict)

    def __hash__(self):
        if self._hash is None:
            self._hash = hash(frozenset(self._dict.items()))
        return self._hash

    def __eq__(self, other):
        return self._dict == other._dict


# From http://code.activestate.com/recipes/576694/
class OrderedSet(collections.MutableSet):

    def __init__(self, iterable=None):
        self.end = end = []
        end += [None, end, end]         # sentinel node for doubly linked list
        self.map = {}                   # key --> [key, prev, next]
        if iterable is not None:
            self |= iterable

    def __len__(self):
        return len(self.map)

    def __contains__(self, key):
        return key in self.map

    def add(self, key):
        """
        Add a new member to the set. Return a :class:`bool` indicating
        if the object was added (``False`` indicates that the object
        was already in the set.
        """
        added = False
        if key not in self.map:
            end = self.end
            curr = end[1]
            curr[2] = end[1] = self.map[key] = [key, curr, end]
            added = True
        return added

    def discard(self, key):
        if key in self.map:
            key, prev, next = self.map.pop(key)
            prev[2] = next
            next[1] = prev
        else:
            key = None
        return key

    def __iter__(self):
        end = self.end
        curr = end[2]
        while curr is not end:
            yield curr[0]
            curr = curr[2]

    def __reversed__(self):
        end = self.end
        curr = end[1]
        while curr is not end:
            yield curr[0]
            curr = curr[1]

    def pop(self, last=True):
        if not self:
            raise KeyError('set is empty')
        key = self.end[1][0] if last else self.end[2][0]
        self.discard(key)
        return key

    def __repr__(self):
        if not self:
            return '%s()' % (self.__class__.__name__,)
        return '%s(%r)' % (self.__class__.__name__, list(self))

    def __eq__(self, other):
        if isinstance(other, OrderedSet):
            return len(self) == len(other) and list(self) == list(other)
        return set(self) == set(other)


class FifoSequence(collections.Sequence):
    """
    A sequence that has a fixed length and removes items (FIFO)
    when an arbitrary length is reached.
    """

    @property
    def max_length(self):
        return self.__max_length

    def __init__(self, max_length, initial__items=None):
        self.__max_length = max_length
        self.__items = initial__items or []

    def append(self, item):
        if len(self) > self.__max_length:
            self.__items = self.__items[1:]
        self.__items.append(item)

    def __getitem__(self, index):
        return self.__items.__getitem__(index)

    def __index__(self, item):
        return self.__items.__index__(item)

    def __len__(self):
        return len(self.__items)

    def __contains__(self, item):
        return self.__items.__contains__(item)


def dictmerge(target, source, *sources):
    """
    Recursively merge `targets` into `source`.
    """
    if sources:
        dictmerge(target, source)
        for source in sources:
            dictmerge(target, source)
    else:
        if not isinstance(source, dict):
            return source
        for k, v in source.iteritems():
            if k in target and isinstance(target[k], dict):
                dictmerge(target[k], v)
            else:
                target[k] = copy.deepcopy(v)
    return target



def is_sequence(obj):
    return hasattr(obj, '__iter__')



class Array(collections.Sequence):

    def __init__(self, item_type, items=None, **params):
        self._items = tuple(map(lambda x: item_type(x, **params),
            items or []))

    def __getitem__(self, index):
        return self._items.__getitem__(index)

    def __index__(self, item):
        return self._items.__index__(item)

    def __len__(self):
        return len(self._items)

    def __contains__(self, item):
        return self._items.__contains__(item)


class SerializableArray(Array):

    def serialize(self, serializer, **serializer_kwargs):
        return serializer(map(lambda x: dict(x), self._items))