import unittest

from libtng import entity


__all__ = ['EntityFieldTestCase']


class TestEntity(entity.Entity):
    integer_property    = entity.IntegerProperty()
    public_integer_prop = entity.IntegerProperty(public_setter=True, public_getter=True)


class EntityFieldTestCase(unittest.TestCase):
    do = TestEntity
    public_attname  = 'public_integer_prop'
    private_attname = 'integer_property'

    def setUp(self):
        self.opts = self.do._meta

    def test_integerfield1(self):
        assert hasattr(self.do, self.public_attname)

    def test_field_in_meta(self):
        public_attnames = [x.public_attname for x in self.opts.props]
        assert self.public_attname in public_attnames, public_attnames

    def test_init(self):
        instance = self.do()
        self.assertTrue(getattr(instance, self.public_attname) is None)

    def test_settattr1(self):
        instance = self.do()
        value = 1
        setattr(instance, self.public_attname, value)

    def test_settattr2(self):
        instance = self.do()
        value = 1
        setattr(instance, self.public_attname, value)
        self.assertEquals(getattr(instance, self.public_attname), value)

    def test_settattr_private(self):
        instance = self.do()
        value = 1
        f = lambda: setattr(instance, self.private_attname, value)
        self.assertRaises(AttributeError, f)

    def test_has_state(self):
        instance = self.do()
        self.assertTrue(hasattr(instance,'_state'))

    def test_validate_state(self):
        validator = self.opts.validator
        instance = self.do(
            integer_property=1,
            public_integer_prop=1
        )
        self.assertTrue(validator.is_valid_state(instance._state))

    def test_validate_state_on_init(self):
        """
        Entity.create() validates input and raises an exception
        when the input would create an invalid state.
        """
        validator = self.opts.validator
        self.assertRaises(entity.InvalidStateError, self.do.create,
            integer_property=1, public_integer_prop=None)

    def test_state(self):
        """
        State must be unique for each instance.
        """
        i1 = TestEntity()
        i2 = TestEntity()
        self.assertNotEquals(i1._state, i2._state)

if __name__ == '__main__':
    unittest.main()