"""
Framework for the validation of instance attributes.
"""
from bisect import bisect
import collections
import sys
import re
import abc
import copy
import functools
import types


from libtng import six
from libtng.functional import cached_property
from libtng.exceptions import ImproperlyConfigured
from libtng.meta import accepts_keyword_arguments
from libtng.translation import string_concat

from validators import EntityStateValidator
from state import EntityState
import props
import ident

__all__ = [
    'DomainObject',
    'Entity',
    'BaseEntity',
    'EntityMeta',
    'EntityBaseMeta',
    'privatemethod',
    'PrivateMethod'
]

DEFAULT_NAMES = ['validator','abstract','immutable','unique','protected_attributes']


class NoIdentityError(TypeError):
    pass


# Calculate the verbose_name by converting from InitialCaps to "lowercase with spaces".
get_verbose_name = lambda class_name: re.sub('(((?<=[a-z])[A-Z])|([A-Z](?![A-Z]|$)))', ' \\1', class_name).lower().strip()


def privatecaller(func):
    """
    Decorator that set an `is_private_caller` attribute on the
    :class:`~libtng.entity.state.EntityState` of an instance,
    so that methods can access the private properties; and sets
    the attribute to ``False`` upon returning.

    Note the `is_private_caller` status is set to the thread invoking
    the instance method, so the private attribute becomes exposed to
    all functions within the wrapped function.
    """
    is_private = getattr(func, 'is_private', False)
    @functools.wraps(func)
    def wrapper(self, *args, **kwargs):
        initial = self._state.is_private_caller()
        if is_private and not initial:
            raise PrivateMethod("{0} is a private method.".format(func.__name__))
        try:
            self._state.set_private_caller(True)
            retval = func(self, *args, **kwargs)
        finally:
            self._state.set_private_caller(initial)
        return retval
    return wrapper


def privatemethod(func):
    """
    Decorator that marks a method as private.
    """
    setattr(func, 'is_private', True)
    return func


class PrivateMethod(Exception):
    pass


class Options(object):
    """
    Describes the structure and behavior of a :class:`Entity`.

    :ivar unique:
        indicates that the entity should be unique; the meaning of unique
        is is determined by it's domain.
    """

    def __init__(self, meta, **kwargs):
        self.local_props = []
        self.local_many_to_many = []
        self.virtual_props = []
        self.parents = []
        self.meta = meta
        self.validator = None
        self.ident = None
        self.verbose_name_plural = None
        self.abstract = False
        self.immutable = False
        self.unique = False

    def contribute_to_class(self, cls, name):
        cls._meta = self
        self.entity = cls
        self.object_name = cls.__name__
        self.entity_name = self.object_name.lower()
        self.verbose_name = get_verbose_name(self.object_name)

        # Next, apply any overridden values from 'class Meta'.
        if self.meta:
            meta_attrs = self.meta.__dict__.copy()
            for name in self.meta.__dict__:
                # Ignore any private attributes that Django doesn't care about.
                # NOTE: We can't modify a dictionary's contents while looping
                # over it, so we loop over the *original* dictionary instead.
                if name.startswith('_'):
                    del meta_attrs[name]
            for attr_name in DEFAULT_NAMES:
                if attr_name in meta_attrs:
                    setattr(self, attr_name, meta_attrs.pop(attr_name))
                elif hasattr(self.meta, attr_name):
                    setattr(self, attr_name, getattr(self.meta, attr_name))

            # verbose_name_plural is a special case because it uses a 's'
            # by default.
            if self.verbose_name_plural is None:
                self.verbose_name_plural = string_concat(self.verbose_name, 's')

            # Any leftover attributes must be invalid.
            if meta_attrs != {}:
                raise TypeError("'class Meta' got invalid attribute(s): %s" % ','.join(meta_attrs.keys()))
        else:
            self.verbose_name_plural = string_concat(self.verbose_name, 's')
        del self.meta

        # Construct the entity validator class
        self.validator = self.construct_validator()

    def construct_validator(self):
        """
        Create a :class:`libtng.entity.validators.EntityStateValidator`
        subclass responsible for validating the entity state.
        """
        bases = (EntityStateValidator,) if not self.validator\
            else (self.validator,)
        attrs = {
            'meta'      : self,
            'entity'    : self.entity,
            'props'     : self.props,
            'prop_dict' : self.prop_dict
        }
        return type(self.object_name + 'StateValidator', bases, attrs)()


    def get_local_props(self):
        """
        Get all local props declared on a :class:`DomainObject`
        subclass.
        """
        return self.local_props + \
            self.local_many_to_many + \
            self.virtual_props

    @cached_property
    def props(self):
        """
        The getter for self.props. This returns the list of prop objects
        available to this model (including through parent models).

        Callers are not permitted to modify this list, since it's a reference
        to this instance (not a copy).
        """
        try:
            self._prop_name_cache
        except AttributeError:
            self._fill_props_cache()
        return self._prop_name_cache

    @property
    def prop_dict(self):
        return {x.name: x for x in self.props}

    def add_prop(self, prop):
        self.local_props.insert(bisect(self.local_props, prop), prop)
        if hasattr(self, '_prop_cache'):
            del self._prop_cache
            del self._prop_name_cache
            # The props, concrete_props and local_concrete_props are
            # implemented as cached props for performance reasons.
            # The attrs will not exists if the cached property isn't
            # accessed yet, hence the try-excepts.
            try:
                del self.props
            except AttributeError:
                pass
            try:
                del self.concrete_props
            except AttributeError:
                pass
            try:
                del self.local_concrete_props
            except AttributeError:
                pass

        if hasattr(self, '_name_map'):
            del self._name_map

    def _fill_props_cache(self):
        cache = []
        for parent in self.parents:
            for prop, model in parent._meta.get_props_with_model():
                if model:
                    cache.append((prop, model))
                else:
                    cache.append((prop, parent))
        cache.extend([(f, None) for f in self.local_props])
        self._prop_cache = tuple(cache)
        self._prop_name_cache = list(sorted([x for x, _ in cache], key=lambda x: x.creation_counter))

    def get_prop(self, prop_name):
        return self.prop_dict[prop_name]

    def get_prop_names(self):
        """
        Return a list containing the names of all properties, in the order
        that they were declared on the entity.
        """
        return [x.name for x in self.props]

    def get_identity_props(self):
        """
        Return a :class:`list` holding the properties that make up
        the identity of an entity.
        """
        return [x for x in self.props if x.identity]

    def has_identity(self):
        """
        Return a :class:`bool` indicating if the identity has specified
        an identity.
        """
        return self.identity is not None

    def set_identity(self):
        if bool(self.get_identity_props()):
            name = self.object_name + 'Identity'
            bases = (ident.EntityIdentity,)
            attrs = {
                'entity_class'  : self.entity,
                'meta'          : self,
                'ident_props'   : self.get_identity_props()
            }
            self.identity = type(name, bases, attrs)


class EntityBaseMeta(type):

    def add_to_class(cls, name, value):
        if hasattr(value, 'contribute_to_class'):
            value.contribute_to_class(cls, name)
        elif isinstance(value, types.FunctionType): # Before a class is created, it's methods are FunctionType
            setattr(cls, name, privatecaller(value))
        else:
            setattr(cls, name, value)

    def getmeta(self, *args, **kwargs):
        return Options(*args, **kwargs)

    def __new__(cls, name, bases, attrs):
        super_new = super(EntityBaseMeta, cls).__new__

        # six.withmetaclass() inserts an extra class called 'NewBase' in the
        # inheritance tree: Model -> NewBase -> object. But the initialization
        # should be executed only once for a given model class.

        # attrs will never be empty for classes declared in the standard way
        # (ie. with the `class` keyword). This is quite robust.
        if name == 'NewBase' and attrs == {}:
            return super_new(cls, name, bases, attrs)

        # Also ensure initialization is only performed for subclasses of Model
        # (excluding Model class itself).
        parents = [b for b in bases if isinstance(b, EntityBaseMeta) and
                not (b.__name__ == 'NewBase' and b.__mro__ == (b, object))]
        if not parents:
            return super_new(cls, name, bases, attrs)

        # Create the class
        module = attrs.pop('__module__')
        new_class = super_new(cls, name, bases, {'__module__': module})
        attrmeta = attrs.pop('Meta', None)
        abstract = getattr(attrmeta, 'abstract', False)
        if not attrmeta:
            meta = getattr(new_class, 'Meta', None)
        else:
            meta = attrmeta
        basemeta = getattr(new_class, 'meta', None)
        new_class.add_to_class('_meta', new_class.getmeta(meta))

        # Add all attributes to the class.
        for obj_name, obj in attrs.items():
            new_class.add_to_class(obj_name, obj)

        # All the props of any type declared on this model
        new_props = new_class._meta.get_local_props()
        prop_names = set([f.name for f in new_props])

        # Add props from parents
        for base in parents:
            original_base = base
            if not hasattr(base, '_meta'):
                # Things without meta aren't functional models, so they're
                # uninteresting parents.
                continue

            parent_props = base._meta.local_props
            # Check for clashes between locally declared props and those
            # on the base classes (we cannot handle shadowed props at the
            # moment).
            for prop in parent_props:
                if prop.name in prop_names:
                    raise ImproperlyConfigured(
                        "Local prop {0} in class {1} clashes with fie"
                        "ld of similar name from base class {2}"\
                        .format(prop.name, name, base.__name__)
                    )

            # Add fields from parent.
            for prop in parent_props:
                new_class.add_to_class(prop.name, copy.deepcopy(prop))

        # Construct the identity if the entity has specified
        # one.
        new_class._meta.set_identity()

        # Run hooks
        new_class._meta.namedtuple = new_class.get_namedtuple(new_class._meta)

        return new_class

    def get_namedtuple(self, meta):
        return collections.namedtuple(meta.object_name + 'Parameters',
            [x for x in meta.get_prop_names()])


class EntityMeta(EntityBaseMeta, abc.ABCMeta):
    pass


class BaseEntity(object):
    pass


class Entity(six.with_metaclass(EntityMeta)):

    @classmethod
    def create(cls, *args, **kwargs):
        self = cls()
        state = self._state
        # Set props from kwargs
        for prop in self._meta.props:
            value = kwargs.pop(prop.name, None)
            if value is None and prop.default_on_init:
                value = prop.default if not callable(prop.default)\
                    else prop.default(self, *args, **kwargs)
            state.set_prop(prop.name, value)
        state.validate()

        if kwargs:
            raise TypeError(kwargs)
        return self

    def __new__(cls, *args, **kwargs):
        self = super(Entity, cls).__new__(cls)
        state = self._state = EntityState(self)

        # Input arguments to __init__ are assumed valid.
        for prop in self._meta.props:
            state.set_prop(prop.name, kwargs.get(prop.name, prop.default))

        # Run custom init
        assert accepts_keyword_arguments(self.init),\
            "init() must accept keyword arguments."
        self.init(*args, **kwargs)
        return self

    def init(self, *args, **kwargs):
        pass

    @privatemethod
    def as_dict(self):
        t = self.as_tuple()
        return {x: getattr(t, x) for x in t._fields}

    @privatemethod
    def as_tuple(self):
        return self._meta.namedtuple(**{x: self._state.get_prop(x)
            for x in self._meta.get_prop_names()})

    def get_identity(self):
        """
        Return a :class:`~libtng.entity.ident.EntityIdentity`
        subclass representing the conceptual identity of the
        object.
        """
        opts = self._meta
        if not opts.has_identity():
            raise NoIdentityError("{0} has not declared an identity."\
                .format(opts.object_name))
        return opts.identity(self._state)

    def is_unique(self):
        """
        Return a boolean indicating if the instance is unique,
        based on it's identity.
        """
        return self._meta.unique

    def __eq__(self, other):
        return isinstance(other, self.__class__) \
            and self.get_identity() == other.get_identity()

    def __ne__(self, other):
        return not self.__eq__(other)

    def __hash__(self):
        return hash(self.get_identity())

    def __repr__(self):
        opts = self._meta
        if not opts.has_identity():
            value = super(Entity, self).__repr__()
        else:
            ident = ', '.join([str(x) for x in self.get_identity()])
            value = '{0}({1})'.format(opts.object_name, ident)
        return value