from collections import Mapping, Set
import copy

from libtng import six
from libtng.functional import cached_property


DEFAULT_SLOTS = ['entity_class','meta','ident_props','state']



class EntityPropertyDescriptor(object):

    def __init__(self, prop_name):
        self.prop_name = prop_name

    def __get__(self, instance, cls):
        if instance:
            return instance.state.get_prop(self.prop_name)


class EntityIdentityMetaclass(type):

    def __new__(cls, name, bases, attrs):
        super_new = super(EntityIdentityMetaclass, cls).__new__

        # six.withmetaclass() inserts an extra class called 'NewBase' in the
        # inheritance tree: EntityIdentity -> NewBase -> object. But the initialization
        # should be executed only once for a given model class.

        # attrs will never be empty for classes declared in the standard way
        # (ie. with the `class` keyword). This is quite robust.
        if name == 'NewBase' and attrs == {}:
            return super_new(cls, name, bases, attrs)

        # Also ensure initialization is only performed for subclasses of
        # EntityIdentity (excluding EntityIdentity class itself).
        parents = [b for b in bases if isinstance(b, EntityIdentityMetaclass) and
                not (b.__name__ == 'NewBase' and b.__mro__ == (b, object))]
        if not parents:
            return super_new(cls, name, bases, attrs)

        # Create descriptors for the properties and add them to slots.
        slots = copy.deepcopy(DEFAULT_SLOTS)
        for prop in attrs['ident_props']:
            slots.append(prop.name)
            attrs[prop.name] = EntityPropertyDescriptor(prop.name)

        return super_new(cls, name, bases, attrs)


class EntityIdentity(six.with_metaclass(EntityIdentityMetaclass)):
    """
    Represents the conceptual identity of an entity.
    """

    def __init__(self, state):
        self.state = state

    def get_props(self):
        return tuple([getattr(self, x.name) for x in self.ident_props])

    def __eq__(self, other):
        return isinstance(other, self.__class__) \
            and self.get_props() == other.get_props()

    def __ne__(self, other):
        return not self.__eq__(other)

    def __repr__(self):
        return "{0}Identity({1})".format(self.meta.object_name,
            ', '.join([str(x) for x in self.get_props()]))

    def __hash__(self):
        return hash(self.get_props())

    def __iter__(self):
        return iter(self.get_props())