from libtng.exceptions import ValidationError
from libtng import validators


class NO_VALUE:
    pass


class PrivateAttribute(AttributeError):
    pass


class ReadOnlyAttribute(AttributeError):
    pass


class InvalidStateError(ValidationError):
    pass


class PrivateGetter(object):

    def __init__(self, field):
        self.field = field

    def __get__(self, instance, cls):
        if instance is None:
            return self.field
        return instance._state.get_prop(self.field.name)


class DomainPropertyDescriptor(property):
    """
    Descriptor class to access a field value on a
    :class:`~libtouchngo.ddd.DomainObject` instance.
    """

    def __init__(self, field):
        self.field = field
        self.entity = field.entity
        self.private = (not field.public_setter and not field.public_getter)
        self.readonly = not field.public_getter
        self.public_attname = field.public_attname
        self.private_attname = field.private_attname
        property.__init__(self, self.getter, self.setter)

    def getter(self, instance):
        if self.private and not instance._state.is_private_caller():
            raise PrivateAttribute("{0}.{1} is a private attribute.".format(
                self.entity.__name__, self.public_attname))
        return self.field.to_python(instance._state.get_prop(self.field.name))

    def setter(self, instance, value):
        if self.readonly and not instance._state.is_private_caller():
            raise ReadOnlyAttribute("{0}.{1} is a read only attribute.".format(
                self.entity.__name__, self.public_attname))
        state = instance._state
        old = state.get_prop(self.field.name) or NO_VALUE
        new = self.field.get_prep_value(instance, old, value)
        cleaned = self.field.clean(new, old=old, instance=instance, state=state)
        instance._state.set_prop(self.field.name, self.field.clean(new))

class DomainProperty(object):
    """
    Base class for all field types.
    """
    descriptor_class = DomainPropertyDescriptor
    empty_strings_allowed = True
    empty_values = validators.EMPTY_VALUES + [NO_VALUE]

    # These track each time a DomainProperty instance is created. Used to retain order.
    # The auto_creation_counter is used for fields that Django implicitly
    # creates, creation_counter is used for all user-specified fields.
    creation_counter = 0
    auto_creation_counter = -1
    default_validators = [] # Default set of validators
    default_error_messages = {
        'invalid_choice': 'Value %(value)r is not a valid choice.',
        'null'          : 'This property cannot be None.',
        'blank'         : 'This property cannot be empty on initialization.'
    }

    @property
    def default(self):
        return self._default if (self._default != NO_VALUE)\
            else None

    def __init__(self, identity=False, name=None, verbose_name=None, db_column=None,
        unique=False, default=NO_VALUE, null=False, deferred=False, auto_created=False,
        choices=None, clean_on_assignment=True, max_length=None, validators=None,
        editable=True, blank=False, error_messages=None, immutable=True, uselist=None,
        is_serial=False, public_setter=False, public_getter=False, required=True,
        default_on_init=False):
        """
        Instantiates a new :class:`DomainProperty` instance.

        :param identity:
            specifies if the field is (part of) the objects' identity. The data type
            of the property must be hashable if ``identity=True``.
        :type identity: :class:`bool`
        :param name:
            the name of the attribute on the created class.
        :param verbose_name:
            a verbose name to display in an application.
        :param unique:
            indicates if the value should be unique in the table.
        :param default:
            specifies a default value.
        :param blank:
            indicates if the value may be omitted on instance
            initialization.
        :param null:
            indicates if the field may hold a None (NULL) value.
        :param deferred:
            specifies if the column is lazy-loaded from the database.
        :param auto_created:
            indicates if the field is auto created.
        :param choices:
            an iterable of key/value pairs specifying allowed choices.
        :param clean_on_assignment:
            specifies if a field should be cleaned immediatly on assigment.
        :param max_length:
            specifies the maximum length.
        :param validators:
            a list of additional validators.
        :param editable:
            indicates if a field is editable through a user-interface.
        :param blank:
            indicates if the field may be blank upon submission through a user-interface.
        :param error_messages:
            a dictionary containing additional error messages.
        :param immutable:
            indicates if the field is immutable after initial assigment.
        :param is_serial:
            indicates if the column generates a default value using a
            generator provided by the database engine.
        :param private:
            indicates if the attribute is private. Default is True.
        :param default_on_init:
            indicates if the default value of the property may be used
            on instance initialization if missing.
        """
        self.identity = identity
        self.name = name
        self.verbose_name = verbose_name
        self.db_column = db_column
        self.unique = unique
        self._default = default
        self.null = null
        self.deferred = deferred
        self.auto_created = auto_created
        if auto_created:
            self.creation_counter = DomainProperty.auto_creation_counter
            DomainProperty.auto_creation_counter -= 1
        else:
            self.creation_counter = DomainProperty.creation_counter
            DomainProperty.creation_counter += 1
        self._choices = choices or []
        self.choices = self._choices
        self.clean_on_assignment = clean_on_assignment
        self.max_length = max_length
        self.validators = self.default_validators + list(validators or [])
        self.editable = editable
        self.blank = blank
        messages = {}
        for c in reversed(self.__class__.__mro__):
            messages.update(getattr(c, 'default_error_messages', {}))
        messages.update(error_messages or {})
        self.error_messages = messages
        self.immutable = immutable
        self.is_serial = is_serial
        self.public_getter = public_getter
        self.public_setter = public_setter
        self.required = required
        self.default_on_init = default_on_init

    def is_empty(self, value):
        """
        Return a boolean indicating if the property considers the `value`
        empty.
        """
        return value in self.empty_values

    def get_attname(self):
        return self.name

    def get_attname_column(self):
        attname = self.get_attname()
        private_attname = '_{0}'.format(attname)
        return attname, private_attname

    def set_attributes_from_name(self, name):
        if not self.name:
            self.name = name
        self.public_attname, self.private_attname = self.get_attname_column()
        if self.verbose_name is None and self.name:
            self.verbose_name = self.name.replace('_', ' ')

    def contribute_to_class(self, cls, name, virtual_only=False):
        """
        Contributes the field to a :class:`libtouchngo.ddd.domain.DomainEntity`
        subclass. Sets the `attname` and `entity` attributes on the
        field.
        """
        self.set_attributes_from_name(name)
        self.entity = cls
        if virtual_only:
            cls._meta.add_virtual_field(self)
        else:
            cls._meta.add_prop(self)
        if self.choices:
            warnings.warn("TODO: Implement choices for DomainProperty.")
        setattr(cls, self.public_attname, DomainPropertyDescriptor(self))
        setattr(cls, self.private_attname, PrivateGetter(self))

    def to_python(self, new):
        return new

    def get_prep_value(self, instance, old, new):
        return new

    def clean(self, value, old=None, instance=None, state=None, init=False):
        """
        Convert the value's type and run validation. Validation errors
        from to_python and validate are propagated. The correct value is
        returned if no error is raised.
        """
        value = self.to_python(value)
        self.validate(value, instance)
        self.run_validators(value)
        return value

    def validate(self, value, model_instance):
        """
        Validates value and throws ValidationError. Subclasses should override
        this to provide validation logic.
        """
        if not self.editable:
            # Skip validation for non-editable fields.
            return

        if self._choices and value not in self.empty_values:
            for option_key, option_value in self.choices:
                if isinstance(option_value, (list, tuple)):
                    # This is an optgroup, so look inside the group for
                    # options.
                    for optgroup_key, optgroup_value in option_value:
                        if value == optgroup_key:
                            return
                elif value == option_key:
                    return
            raise InvalidStateError(
                self.error_messages['invalid_choice'],
                code='invalid_choice',
                params={'value': value},
            )

        if (value is None and not self.null) and self.required:
            raise InvalidStateError(self.error_messages['null'], code='null')

        if self.required and (value in self.empty_values):
            raise InvalidStateError(self.error_messages['blank'], code='blank')

    def run_validators(self, value):
        if value in self.empty_values:
            return

        errors = []
        for v in self.validators:
            try:
                v(value)
            except InvalidStateError as e:
                if hasattr(e, 'code') and e.code in self.error_messages:
                    e.message = self.error_messages[e.code]
                errors.extend(e.error_list)

        if errors:
            raise InvalidStateError(errors)




class IntegerProperty(DomainProperty):
    python_type = int


class StringProperty(DomainProperty):
    python_type = unicode
