"""
Exposes a base class for long running processes.
"""
import time
import signal


class BaseProcess(object):

    def __init__(self, setupfunc=None, suppress_exceptions=False, framerate=None):
        """Initialize a new :class:`BaseProcess` instance.

        Args:
            setupfunc: a callable responsible for setting up the environment
                before entering the main event loop.
            suppress_exceptions (bool): indicates if exceptions are to be
                suppressed. Default is False, meaning that exceptions will
                be raised after cleaning up.
            framerate: specifies a waiting period after finished one
                event loop.
        """
        self._needs_update = True
        self._must_exit = False
        self._setupfunc = setupfunc
        self._suppress_exceptions = suppress_exceptions
        self._framerate = framerate

    def setup(self):
        """Hook to set up the process state."""
        pass

    def run(self):
        """Enter the process main loop and execute the
        :func:`BaseProcess.main_event_loop`.
        """

        # Setup the process.
        self.setup()

        # If a setup callable has been passed to the process
        # constructor.
        if self._setupfunc is not None:
            self._setupfunc(self)

        exception = None
        while True:

            # Check if we need to exit and if so bail out
            # immediately.
            if self._must_exit:
                self.do_cleanup(True)
                self.do_exit()
                break

            # if the update() method has been called,
            # refresh the state of the process.
            if self._needs_update:
                self._do_update()

            try:
                self.main_event()
            except NotImplementedError:
                raise
            except Exception as exception:
                if self.exception_handler(exception):
                    self.do_cleanup(False)
                    raise


            # If a framerate was specified, sleep until
            # passed.
            if self._framerate is not None:
                time.sleep(self._framerate)

    def main_event(self):
        raise NotImplementedError

    def update(self):
        """Indicates that :meth:`BaseProcess.do_update`
        should be called."""
        self._needs_update = True

    def join(self):
        """Gracefully exits the process."""
        self._must_exit = True

    def _do_update(self):
        try:
            self.do_update()
        finally:
            self._needs_update = False

    def do_update(self):
        """Hook to update the program state."""
        pass

    def do_exit(self):
        """Gracefully exit the process. May perform any cleanup
        tasks needed by the process."""
        pass

    def do_cleanup(self, gracecul):
        """Performs cleanup prior to exiting (wether it's gracecul or
        caused by an exception).

        Args:
            graceful: indicates if the cleanup is the result of a
                graceful main event loop interruption.

        Returns:
            None
        """
        pass

    def _signal_handler(self, signum, frame):
        if signum == signals.SIGHUP:
            self.update()

    def signal_handler(self, signum, frame):
        pass

    def exception_handler(self, exception):
        """Hook to handle an exception in the main event loop.
        MUST return a boolean indicating if the exception should
        be reraised.
        """
        return True