from os.path import dirname
import sys
import os
import collections

from ansible.runner import Runner
from ansible.inventory import Inventory
from ansible.inventory.host import Host
from ansible.inventory.group import Group
from ansible import utils
from ansible import errors
import ansible.playbook
import ansible.constants as C

import callbacks

C.HOST_KEY_CHECKING = False


class QueryParser(object):

    def __init__(self, query):
        self.query = query
        self.host_vars_from_top = None
        self.groups = self._parse(query)

    def _parse(self, query):
        self.raw = query.as_inventory()
        all_hosts = {}
        group_all= Group('all')
        groups = dict(all=group_all)
        group = None
        for (group_name, data) in self.raw.items():
            if group_name == '_meta':
                if 'hostvars' in data:
                    self.host_vars_from_top = data['hostvars']
                    continue
            if group_name != group_all.name:
                group = groups[group_name] = Group(group_name)
            else:
                group = group_all
            host = None

            if not isinstance(data, dict):
                data = {'hosts': data}
            elif not any(k in data for k in ('hosts','vars')):
                data = {'hosts': [group_name], 'vars': data}

            if 'hosts' in data:
                for hostname in data['hosts']:
                    if not hostname in all_hosts:
                        all_hosts[hostname] = Host(hostname)
                    host = all_hosts[hostname]
                    group.add_host(host)

            if 'vars' in data:
                for k, v in data['vars'].iteritems():
                    if group.name == group_all.name:
                        group_all.set_variable(k, v)
                    else:
                        group.set_variable(k, v)
            if group.name != group_all.name:
                group_all.add_child_group(group)

        # Separate loop to ensure all groups are defined
        for (group_name, data) in self.raw.items():
            if group_name == '_meta':
                continue
            if isinstance(data, dict) and 'children' in data:
                for child_name in data['children']:
                    if child_name in groups:
                        groups[group_name].add_child_group(groups[child_name])
        return groups

    def get_host_variables(self, host):
        """
        Returns the variables for a specific host.
        """
        if self.host_vars_from_top is not None:
            got = self.host_vars_from_top.get(host.name, {})
            return got
        raise NotImplementedError


class Inventory(Inventory):
    """
    Host inventory for ansible using a using a :class:`sqlalchemy.query.Query`
    subclass. The subclass must expose an ``as_inventory()()`` method, returning
    a :class:`dict` as described in http://docs.ansible.com/developing_inventory.html.
    """

    def __init__(self, query, base_dir=None):
        self.__base_dir = base_dir
        self.host_list = None
        self.query = query
        self._vars_per_host  = {}
        self._vars_per_group = {}
        self._hosts_cache    = {}
        self._groups_list    = {}
        self._pattern_cache  = {}
        self._playbook_basedir = None
        self.groups = []
        self._restriction = None
        self._also_restriction = None
        self._subset = None
        self.parser = QueryParser(query)
        self.groups = self.parser.groups.values()
        utils.plugins.vars_loader.add_directory(self.basedir(), with_subdir=True)
        self._vars_plugins = [ x for x in utils.plugins.vars_loader.all(self) ]

    def basedir(self):
        return self.__base_dir


FakeOptions = collections.namedtuple('FakeOptions', ['module_name','tree','one_line'])


class Runner(Runner):

    @classmethod
    def fromoptions(cls, inventory, module_name, base_dir=None, verbosity=0, *args, **kwargs):
        inventory = Inventory(inventory, base_dir)
        utils.VERBOSITY = 0
        while utils.VERBOSITY <= verbosity:
            utils.increment_debug(None,None,None,None)
        stats = callbacks.AggregateStats()
        runner_cb = callbacks.CliRunnerCallbacks()
        runner_cb.options = FakeOptions(module_name=module_name, tree=None,
            one_line=False)
        extra_vars = kwargs.pop('extra_params', {})
        kwargs.update({
            'callbacks': runner_cb,
            'private_key_file': kwargs.pop('private_key', None)
        })
        if 'any_errors_fatal' in kwargs:
            del kwargs['any_errors_fatal']
        return cls(inventory=inventory, module_name=module_name, *args, **kwargs)

    def run(self, *args, **kwargs):
        result = super(Runner, self).run(*args, **kwargs)
        items = []
        for host, raw_data in result['dark'].items():
            items.append(callbacks.TaskResult(host, raw_data, unreachable=True))
        for host, raw_data in result['contacted'].items():
            items.append(callbacks.TaskResult(host, raw_data, unreachable=False))
        return items



class PlayBook(ansible.playbook.PlayBook):

    @classmethod
    def fromoptions(cls, inventory, playbook, base_dir=None,
        verbosity=0, *args, **kwargs):
        stats = callbacks.AggregateStats()
        inventory = Inventory(inventory, base_dir)
        playbook_cb_cls = kwargs.pop('state_callbacks_class',
            callbacks.PlayBookCallbacks)
        runner_cb_cls = kwargs.pop('runner_callbacks_class',
            callbacks.PlayBookRunnerCallbacks)
        utils.VERBOSITY = 0
        while utils.VERBOSITY <= verbosity:
            utils.increment_debug(None,None,None,None)
        playbook_cb = playbook_cb_cls(verbose=verbosity,
            verbosity=verbosity)
        runner_cb = runner_cb_cls(stats, verbose=verbosity,
            verbosity=verbosity)
        inventory.set_playbook_basedir(dirname(playbook))
        inventory.set_playbook_basedir(dirname(playbook))
        extra_vars = kwargs.pop('extra_params', {})
        kwargs.update({
            'callbacks': playbook_cb,
            'runner_callbacks': runner_cb,
            'stats': stats,
            'extra_vars': extra_vars,
            'private_key_file': kwargs.pop('private_key', None)
        })
        return cls(inventory=inventory, playbook=playbook, *args, **kwargs)

    def run(self, *args, **kwargs):
        summary = super(PlayBook, self).run(*args, **kwargs)
        return callbacks.AggregrateResults(summary, self.stats.results)