"""
Probes a remote host and returns information about it.
"""
import collections

from libtng.formatting import slugify
from libtng.provisioning.commands.base import PreconfiguredModuleCommand
from libtng.provisioning.commands.base import ProvisioningCommandHandler
from libtng.provisioning.commands.base import ProvisioningResult


NetworkDevice = collections.namedtuple('NetworkDevice', ['active','device',
    'ipv4','macaddress','mtu','promisc','type'])
NetworkInterface4 = collections.namedtuple('NetworkInterface4', ['address','netmask','network'])
NetworkInterface6 = collections.namedtuple('NetworkInterface6', ['address','prefix','scope'])


class ProbeResult(ProvisioningResult):
    """
    Holds information about a certain host.
    """

    @property
    def facts(self):
        return self.raw_data['ansible_facts']

    def get_operating_system(self):
        return slugify("{0}-{1}-x{2}".format(
            self.facts['ansible_distribution'],
            self.facts['ansible_distribution_version'].replace('.','-'),
            self.facts['ansible_userspace_bits']
        ))

    def is_vm(self):
        return self.facts.get('ansible_virtualization_role') != 'guest'

    def get_server_type(self):
        """
        Return a :class:`str` holding the server type.
        """
        return 'vm' if self.is_vm() else 'dedicated'

    @property
    def memory(self):
        """
        Return an :class:`int` indicating the memory of the probed
        host, in bytes.
        """
        return self.facts['ansible_memtotal_mb'] * (1024**2)

    @property
    def cpu_count(self):
        """
        An :class:`int` representing the number of CPUs.
        """
        return self.facts['ansible_processor_count']

    @property
    def total_cores(self):
        """
        An :class:`int` representing the total number of cores.
        """
        return self.facts['ansible_processor_cores']

    @property
    def default_ipv4_alias(self):
        """
        A :class:`str` specifying the default IPv4 network interface.
        """
        return self.facts['ansible_default_ipv4']['alias']

    def get_interfaces(self):
        """
        Return a list containing the network interfaces of the
        probed host. The network interfaces are represented as
        dictionaries of the following structure:

        .. code:: python

            # Example network interface.
            {
                "active": True,
                "device": "eth1",
                "ipv4": {
                    "address": "192.168.33.10",
                    "netmask": "255.255.255.0",
                    "network": "192.168.33.0"
                },
                "ipv6": [
                    {
                        "address": "fe80::a00:27ff:feba:5e6c",
                        "prefix": "64",
                        "scope": "link"
                    }
                ],
                "macaddress": "08:00:27:ba:5e:6c",
                "module": "e1000",
                "mtu": 1500,
                "promisc": False,
                "type": "ether"
            }
        """
        interfaces = []
        for alias in self.facts['ansible_interfaces']:
            if alias == 'lo': # Skip the local loopback
                continue
            key = "ansible_{0}".format(alias)
            iface = self.facts[key]
            if iface.get('type') in (None, 'loopback'): # Skip loopback and fake devices
                continue
            if 'ipv4' in iface:
                iface['ipv4'] = NetworkInterface4(**iface.pop('ipv4'))
            else:
                iface['ipv4'] = None
            if 'ipv6' in iface:
                iface.pop('ipv6')
            interfaces.append(NetworkDevice(**iface))
        return interfaces


class ProbeCommand(PreconfiguredModuleCommand):
    module_name = 'setup'


@ProbeCommand.handler
class ProbecommandHandler(ProvisioningCommandHandler):
    result_class = ProbeResult