import types

from libtng.http.exceptions.base import HttpException
from libtng.module_loading import import_by_path
from libtng.exceptions.http import NotFound


class WSGIApplication(object):

    def __init__(self, settings, urls, middleware_classes=None,
        default_mimetype="text/html", request_class=None, logger=None,
        debug=False, response_class=None):
        self.settings = settings
        self.urls = urls
        self.middleware_classes = middleware_classes or []
        self._view_middleware = []
        self._template_response_middleware = []
        self._response_middleware = []
        self._exception_middleware = []
        self._request_middleware = []
        self.default_mimetype = default_mimetype
        self.request_class = request_class
        self.exception_map = {}
        self.load_middleware(middleware_classes or [])
        self._logger = logger
        self._debug = debug
        self.response_class = response_class

    def wsgi_app(self, environ, start_response):
        """Handles an incoming WSGI request.

        Args:
            environ (dict): the WSGI environment.
            start_response: a callable that returns the response to the
                client.

        Returns:
            None
        """
        request = self.request_class(environ)
        try:
            func, endpoint, kwargs = self.urls.match(environ)
            response = self.process_request(request)
            if response is None:
                response = func(request, **kwargs)
        except HttpException as e:
            if self._debug:
                raise
            response = e.render_to_response(request,
                response_factory=self.response_class,
                mimetype=request.preferred_mimetype)
        if response is None:
            msg = "Endpoint '{0}' ({1}) did not return a Response object"\
                .format(endpoint, repr(func))
            raise TypeError(msg)
        return response(environ, start_response)

    def load_middleware(self, middleware_classes):
        """
        Populate middleware lists from :attr:`WSGIApplication.middleware_classes`.
        """
        self._view_middleware = []
        self._template_response_middleware = []
        self._response_middleware = []
        self._exception_middleware = []

        request_middleware = []
        for middleware_path in middleware_classes:
            mw_class = import_by_path(middleware_path)
            try:
                mw_instance = mw_class()
            except MiddlewareNotUsed:
                continue

            if hasattr(mw_instance, 'process_request'):
                request_middleware.append(mw_instance.process_request)
            if hasattr(mw_instance, 'process_view'):
                self._view_middleware.append(mw_instance.process_view)
            if hasattr(mw_instance, 'process_template_response'):
                self._template_response_middleware.insert(0, mw_instance.process_template_response)
            if hasattr(mw_instance, 'process_response'):
                self._response_middleware.insert(0, mw_instance.process_response)
            if hasattr(mw_instance, 'process_exception'):
                self._exception_middleware.insert(0, mw_instance.process_exception)

        # We only assign to this when initialization is complete as it is used
        # as a flag for initialization being complete.
        self._request_middleware = request_middleware

    def process_request(self, request):
        """Runs request middleware on an incoming HTTP request."""
        return self._run_middleware(request, self._request_middleware)

    def _run_middleware(self, request, middleware_callables):
        response = None
        for func in middleware_callables:
            response = func(request)
            if response is not None:
                break
        return response

    def __call__(self, environ, start_response):
        return self.wsgi_app(environ, start_response)
