import collections
import inspect

from libtng import six
from libtng.datastructures import OrderedSet


class EntityCollection(collections.Set):
    type = None

    @classmethod
    def fromtype(cls, entity_type, name):
        """
        Create a new collection class for a domain entity
        `entity_type`.
        """
        attrs = cls._get_entity_methods(entity_type)
        attrs['type'] = entity_type
        return type(name, (cls, ), attrs)

    @classmethod
    def _get_entity_methods(cls, entity_class):
        def create_method(name, method):
            def manager_method(self, *args, **kwargs):
                result = map(lambda x: method(x, *args, **kwargs), self)
                if getattr(method, 'returns_boolean', False):
                    result = all(result)
                return result
            manager_method.__name__ = method.__name__
            manager_method.__doc__ = method.__doc__
            return manager_method

        new_methods = {}
        # Refs http://bugs.python.org/issue1785.
        predicate = inspect.isfunction if six.PY3 else inspect.ismethod
        for name, method in inspect.getmembers(entity_class, predicate=predicate):
            # Only copy missing methods.
            if hasattr(cls, name):
                continue
            # Only copy public methods or methods with the attribute `entity_only=False`.
            entity_only = getattr(method, 'entity_only', None)
            if entity_only or (entity_only is None and name.startswith('_')):
                continue
            # Copy the method onto the manager.
            new_methods[name] = create_method(name, method)
        return new_methods

    def __init__(self, initial=None):
        self._items = []
        map(self.add, initial or [])

    def add(self, instance):
        """Add a new item to the collection."""
        if not isinstance(instance, self.type):
            raise TypeError("Cannot add item of type '{0}' to {1}"\
                .format(type(instance).__name__, type(self).__name__))
        if instance not in self._items:
            self._items.append(instance)

    def __iter__(self):
        return iter(self._items)

    def __len__(self):
        return len(self._items)

    def __contains__(self, item):
        return item in self._items

    def __getitem__(self, key):
        return self._items[key]