import abc

from libtng.exceptions import NOTIMPLEMENTED_SUBCLASS, ProgrammingError
from libtng.cqrs import Command, CommandHandler
from libtng import environ
from libtng.provisioning.provisioner import Provisioner
from libtng.provisioning.collection import ProvisionableCollection
from libtng.provisioning.provisionable import Provisionable

from libtng.provisioning._ansible import Runner, PlayBook


class ProvisioningCommand(Command):
    """
    Abstract base class for provisioning commands.
    """
    _options = None

    def __init__(self, provisionables, options, module_name, module_args,
        playbook_name, extra_params=None):
        """
        Initialize a new :class:`ProvisioningCommand`.

        :param provisionables:
            a :class:`list` of :class:`~libtng.provisioning.Provisionable`
            instances specifying the hosts to run the command on.
        :param options:
            a :class:`~libtng.provisioning.ProvisioningOptions`
            instance.
        :param extra_params:
            a :class:`dict` containing extra context variables to
            provide to the provisioner.

        .. note::

            **Implemenation detail:** the `extra_params` argument is only
            relevant if the :class:`ProvisioningCommand` invokes an
            Ansible playbook. For modules, `extra_params` is ignored.
        """
        if not (bool(playbook_name) ^ bool(module_name)):
            raise ProgrammingError(
                "Provisionable must specify either a module or a playbook.")
        self._provisionables = provisionables
        self._options = options
        self._extra_params = self._playbook_params = extra_params or {}
        self._module_name = module_name
        self._module_args = module_args
        self._playbook_name = playbook_name

    def get_provisionables(self):
        """
        Return a :class:`~libtng.provisioning.ProvisionableCollection`
        of the provisionables affected by the :class:`ProvisioningCommand`.
        Subclasses should override this method for custom provisionable
        initialization.
        """
        return ProvisionableCollection(self._provisionables,
            self._module_name, self._module_args,
            self._playbook_name, self._playbook_params)

    def set_provisioner_options(self, options):
        """
        Assign a :class:`libtng.provisioning.ProvisionerOptions`
        instance specifying the connection and runtime settings.
        """
        if self._options is not None:
            raise RuntimeError("Options already specified.")
        self._options = options

    def get_provisioner_options(self):
        """
        Return a :class:`libtng.provisioning.ProvisionerOptions`
        instance.
        """
        if not self._options:
            raise AttributeError("Options not configured.")
        return self._options

    def get_extra_params(self):
        """
        Return the context variables for the global provisioning
        context.
        """
        return self._extra_params or {}


class PlayBookCommand(ProvisioningCommand):
    playbook_name = None

    def __init__(self, provisionables, options, extra_params=None):
        if self.playbook_name is None:
            raise ProgrammingError(
                "PlayBookCommand subclasses need to declare a `playbook_name` attribute.")
        ProvisioningCommand.__init__(self, provisionables, options,
            None, None, self.playbook_name, extra_params=extra_params)


class ModuleCommand(ProvisioningCommand):

    def __init__(self, provisionables, options, module_name, **module_args):
        ProvisioningCommand.__init__(self, provisionables, options,
            module_name, module_args, None, extra_params=None)



class PreconfiguredModuleCommand(ModuleCommand):
    module_name = None

    def __init__(self, provisionables, options, **module_args):
        super(PreconfiguredModuleCommand, self).__init__(provisionables, options,
            self.module_name, **module_args)


class ProvisioningResult(object):

    @property
    def raw_data(self):
        return self.task_result.raw_data

    @property
    def host(self):
        return self.task_result.host

    @property
    def unreachable(self):
        return self.task_result.unreachable

    def __init__(self, task_result):
        self.task_result = task_result
    def success(self):
        return self.task_result.success()


class ProvisioningCommandHandler(CommandHandler):
    result_class = ProvisioningResult

    def get_provisioner(self, provisioner_class=None):
        """
        Return a :class:`~libtng.provisioning.Provisioner`
        instance with configured with the options specified
        by the :class:`~ProvisioningCommandHandler`.
        """
        Provisioner = provisioner_class or Provisioner
        return Provisioner.fromoptions(self.get_provisioner_options())

    def handle(self, command, *args, **kwargs):
        """
        Execute the command. Receives the command and additional
        parameters passed to the gateway.
        """
        objects = command.get_provisionables()
        assert isinstance(objects, ProvisionableCollection)
        provisioner = Provisioner.fromoptions(command.get_provisioner_options())
        results = provisioner.provision(objects,
            command.get_extra_params())
        try:
            return map(self.result_class, [x for x in results])
        except TypeError as e:
            raise TypeError(results, *e.args)


class SingleHost(Provisionable):
    """
    Represents a single host without any hostvars.
    """

    def get_provisioner_hosts(self):
        return [(self._host, self._groups, self._hostvars)]

    def __init__(self, host, groups, hostvars):
        self._host = host
        self._groups = groups



