import itertools
import copy

from libtng import datastructures
from libtng import six
from libtng.exceptions import ProgrammingError, object_has_no_attribute
from libtng.provisioning.provisionable import Provisionable


class Host(Provisionable):
    """
    Represents a single host on which a playbook or module can be
    run.
    """

    @property
    def hostname(self):
        return self._hostname

    @property
    def hostvars(self):
        v = copy.deepcopy(self._hostvars)
        v[self._name] = self
        return v

    @property
    def groups(self):
        return self._groups

    def __init__(self, hostname, groups=None, hostvars=None, name='host'):
        """
        Initialize a new :class:`Host` instance.

        :param hostname:
            the hostname or IP address identifiying the host.
        :param groups:
            a list of string specifying the groups memberships of
            the :class:`Host`.
        :param hostvars:
            a :class:`dict` containing variables related to the
            :class:`Host`.
        :param name:
            the name of the context-variable holding the :class:`Host`
            instance.
        """
        self._hostname = str(hostname)
        self._groups = tuple(groups or ['all'])
        self._hostvars = hostvars or {}
        self._name = name

    def get_provisioner_hosts(self):
        return [self]

    def add_to_collection(self, collection):
        """
        Add a :class:`Host` instance to a :class:`~libtng.provisioning.HostCollection`.
        """
        collection.add(self._hostname, self)

    def add_to_group(self, group_name):
        self._groups += ('group_name',)

    def __iter__(self):
        return iter([self._hostname, self._groups, self.hostvars])

    def __repr__(self):
        return "Host('{0}')".format(self._hostname)

    def __and__(self, other):
        assert all(map(lambda x: not datastructures.is_sequence(x), self._groups))
        assert all(map(lambda x: not datastructures.is_sequence(x), other._groups))
        if self.hostname != other.hostname:
            raise ProgrammingError("Cannot union different hosts: {0} != {1}."\
                .format(self, other))
        return Host(self.hostname,
            tuple(itertools.chain(self._groups, other.groups)),
            datastructures.dictmerge(self.hostvars, other.hostvars))

    def __hash__(self):
        return hash(self._hostname)

    def __eq__(self, other):
        return hash(self) == hash(other)

    def __getattr__(self, attname):
        try:
            return self._hostvars[attname]
        except KeyError:
            raise object_has_no_attribute(self, attname)