import collections
import itertools

from libtng.datastructures import OrderedSet, is_sequence


class Inventory(collections.Mapping):

    def __init__(self, initial=None):
        self._inventory = collections.defaultdict(OrderedSet, {
            '_meta': {
                'hostvars': collections.defaultdict(dict)
            }
        })
        self._inventory.update(initial or {})
        self._hosts = 0

    def as_inventory(self):
        return self._inventory

    def append(self, host):
        """
        Append a :class:`Host` instance to the inventory.
        """
        hostname, groups, hostvars = host
        for hostname, group in itertools.product([hostname], groups):
            self._inventory[group].add(hostname)
        if hostvars:
            self._inventory['_meta']['hostvars'][hostname].update(hostvars)

    def merge(self, inventory):
        """
        Merges :class:`Inventory` `inventory` into `self`.
        """
        new_inventory = Inventory()
        for host, groups, hostvars in (self + inventory):
            map(lambda x: new_inventory.add_host(x, host), groups)
            new_inventory.set_hostvars(host, hostvars)
        return new_inventory

    def add_host(self, group, host):
        """
        Add a host to the inventory,

        :param group:
            specifies the group to add the host in.
        :param host:
            specifies the hostname or IP-address.
        """
        if host not in self._inventory[group]:
            self._inventory[group].add(host)
            self._hosts +=1

    def set_hostvars(self, host, hostvars):
        """
        Set context variables for a specific host.

        :param host:
            specifies the hostname or IP-address.
        :param hostvars:
            a :class:`dict` containing the variables for the
            specified `host`.
        """
        self._inventory['_meta']['hostvars'][host].update(hostvars)

    def get_group_names(self):
        """
        Return a list containing all group names declared in the
        inventory.
        """
        return [x for x in self._inventory.keys() if not x.startswith('_')]

    def __getitem__(self, key):
        return self._inventory[key]

    def __len__(self):
        return len(self._hosts)

    def __and__(self, other):
        return self.merge(other)

    def __add__(self, other):
        return list(self) + list(other)

    def __iter__(self):
        host_map = collections.defaultdict(lambda: {
            'groups': [],
            'hostvars': {}
        })
        hostvars = self._inventory['_meta']['hostvars']
        for group, hosts in self._inventory.items():
            if group == '_meta':
                continue
            for host in hosts:
                host_map[host]['groups'].append(group)
                host_map[host]['hostvars'].update(hostvars[host])
        for host, params in host_map.items():
            yield host, params['groups'], params['hostvars']