import abc
import collections
import threading
import random
import sys

from libtng import six
from libtng import timezone
from libtng.cqrs.runner import Runner
from libtng.cqrs.gateway.transaction import Transaction
from libtng.cqrs.exceptions import DuplicateCommand


__all__ = [
    'CommandExecution'
]


class IGateway(six.with_metaclass(abc.ABCMeta)):

    def __init__(self, runner=None, transaction_factory=None):
        self._runner = runner or Runner()
        self._unique_commands = collections.defaultdict(set)
        self._transaction_factory = None

    @abc.abstractmethod
    def dispatch(self, command, *args, **kwargs):
        """
        Executes the given command.
        """
        pass

    def transaction(self, command_type, command, metadata):
        if not self._transaction_factory:
            return Transaction()
        return self._transaction_factory(command_type, command, metadata)

    def put(self, command):
        """Put a command an assess if it can be executed; proceed to
        invoke the command-runner to execute. Returns a :class:`libtng.
        cqrs.Transaction` instance representing the transaction.

        Args:
            command (libtng.cqrs.Command): a concrete command implementation.

        Returns:
            libtng.cqrs.Transaction
        """
        tx = self._dispatch(type(command), command, command.metadata)
        return tx

    def _dispatch(self, command_type, command, metadata):
        if hash(command) in self._unique_commands[command_type]:
            raise DuplicateCommand(command, metadata)
        self._unique_commands[command_type].add(hash(command))

        # Create the transaction.
        tx = self.transaction(command_type, command, metadata)
        self.run(command)


        return tx

    def run(self, command):
        command.set_result(self._runner.run(self, command))