from libtng.module_loading import import_string
from libtng.datastructures import OrderedSet
from libtng.dispatch import Signal
from libtng import six


class Publisher(object):
    event_published = Signal(providing_args=['event'])

    def __init__(self):
        self._listeners = OrderedSet()

    def publish(self, event):
        """Publish :class:`~libtng.ddd.event.Event` `event` to all
        listeners.
        """
        for listener in self._listeners:
            listener(self, event)

    def add_listener(self, listener):
        """
        Adds `listener` to the :class:`Publisher`.

        The `listener` argument can be one of the following.

        -   A dictionary containing a ``listener`` member. The remaining
            members will be passed to the listeners constructor.
        -   A string specifying the import path of a listener. The specified
            class will be instantiated without arguments.

        Args:
            listener: specifies the listener and optionally it's constructor
                arguments.

        Returns:
            None
        """
        Listener, kwargs = (import_string(listener), {}) \
            if isinstance(listener, six.string_types)\
            else (import_string(listener.pop('listener')), listener)
        listener = Listener(**kwargs)
        self._listeners.add(listener)



    def update_listeners(self, listeners):
        """Updates the publisher with the specified `listeners`."""
        return map(self.add_listener, listeners)


