"""
Exposes a base class for long running processes.
"""
import sys
import os
import time
import signal
import contextlib


@contextlib.contextmanager
def pidfile(filepath, daemon):
    """Create a pidfile for a process."""
    if not daemon:
        yield
    else:
        try:
            if os.path.exist(filepath):
                print >> sys.stderr, "Pidfile present:", filepath
                sys.exit(-1)
            with open(filepath, 'wb') as f:
                f.write(str(os.getpid()) + '\n')
            yield
        except Exception:
            pass



class BaseProcess(object):
    default_signals = [
        signal.SIGTERM,
        signal.SIGHUP,
        signal.SIGINT
    ]
    signals = []

    def __init__(self, setupfunc=None, suppress_exceptions=False, framerate=None):
        """Initialize a new :class:`BaseProcess` instance.

        Args:
            setupfunc: a callable responsible for setting up the environment
                before entering the main event loop.
            suppress_exceptions (bool): indicates if exceptions are to be
                suppressed. Default is False, meaning that exceptions will
                be raised after cleaning up.
            framerate: specifies a waiting period after finished one
                event loop.
        """
        self._needs_update = True
        self._must_exit = False
        self._setupfunc = setupfunc
        self._suppress_exceptions = suppress_exceptions
        self._framerate = framerate
        self.signals = tuple(
            set(list(self.signals) + list(self.default_signals)))
        map(lambda x: self.signal_handler, self.signals)

    def setup(self):
        """Hook to set up the process state."""
        pass

    def run(self):
        """Enter the process main loop and execute the
        :func:`BaseProcess.main_event_loop`.
        """

        # Setup the process.
        self.setup()

        # If a setup callable has been passed to the process
        # constructor.
        if self._setupfunc is not None:
            self._setupfunc(self)

        exception = None
        while True:
            try:
                # Check if we need to exit and if so bail out
                # immediately.
                if self._must_exit:
                    self.do_cleanup(True)
                    self.do_exit()
                    break

                # if the update() method has been called,
                # refresh the state of the process.
                if self._needs_update:
                    self._do_update()

                try:
                    self.main_event()
                except NotImplementedError:
                    raise
                except Exception as exception:
                    if self.exception_handler(exception):
                        self.do_cleanup(False)
                        if self._must_exit: # Don't raise if we must exit.
                            return
                        raise


                # If a framerate was specified, sleep until
                # passed.
                if self._framerate is not None:
                    time.sleep(self._framerate)
            except KeyboardInterrupt:
                self.join()

    def main_event(self):
        raise NotImplementedError

    def update(self):
        """Indicates that :meth:`BaseProcess.do_update`
        should be called."""
        self._needs_update = True

    def join(self):
        """Gracefully exits the process."""
        self._must_exit = True

    def _do_update(self):
        try:
            self.do_update()
        finally:
            self._needs_update = False

    def do_update(self):
        """Hook to update the program state."""
        pass

    def do_exit(self):
        """Gracefully exit the process. May perform any cleanup
        tasks needed by the process."""
        pass

    def do_cleanup(self, gracecul):
        """Performs cleanup prior to exiting (wether it's gracecul or
        caused by an exception).

        Args:
            graceful: indicates if the cleanup is the result of a
                graceful main event loop interruption.

        Returns:
            None
        """
        pass

    def signal_handler(self, signum, frame):
        if signum == signal.SIGHUP:
            self.update()
        if signum in (signal.SIGTERM, signal.SIGINT):
            self.do_cleanup(signum == signal.SIGINT)
            self._must_exit = True
            self.do_exit()

    def exception_handler(self, exception):
        """Hook to handle an exception in the main event loop.
        MUST return a boolean indicating if the exception should
        be reraised.
        """
        return True
