import abc

from libtng import six
from libtng.exceptions import ValidationError, NON_FIELD_ERRORS

from props import InvalidStateError


class EntityStateValidator(six.with_metaclass(abc.ABCMeta)):
    """
    Validates an entity.

    :ivar entity:
        the :class:`libtng.entity.Entity` this validator
        operates on.
    :ivar props:
        the properties declared on the entity.
    :ivar meta:
        meta-object describing the entity.
    :ivar prop_dict:
        a :class:`dict` mapping property names to
        :class:`libtng.entity.Entity` instances.
    """

    def is_valid_state(self, state, exclude=None, only=None):
        """
        Return a :class:`bool` indicating if the entity is in a
        valid state.
        """
        valid = False
        try:
            self.validate_state(state, exclude=exclude, only=only)
            valid = True
        except InvalidStateError:
            valid = False
        return valid

    def validate_state(self, state, exclude=None, only=None):
        """
        Validates the state of an :class:`libtng.entity.EntityState`
        instance.
        """
        errors = {}
        exclude = set(exclude or [])
        only = set(only or self.meta.get_prop_names())
        to_validate = (set(self.meta.get_prop_names()) & only) - exclude

        try:
            self.clean_props(state, props=to_validate)
        except InvalidStateError as e:
            errors = e.update_error_dict(errors)

        # Form.clean() is run even if other validation fails, so do the
        # same with Model.clean() for consistency.
        try:
            self.clean()
        except InvalidStateError as e:
            errors = e.update_error_dict(errors)
        if errors:
            raise InvalidStateError(errors)

    def clean_props(self, state, props=None):
        """
        Clean the invididual properties on an instance.
        """
        props = props or set(self.meta.get_prop_names())
        errors = {}
        for f in self.meta.props:
            if f.name not in props:
                continue
            # Skip validation for empty fields with blank=True. The developer
            # is responsible for making sure they have a valid value.
            raw_value = state.get_prop(f.name)
            # if not f.required and raw_value in f.empty_values:
            #     continue
            try:
                cleaned = f.clean(raw_value, self)
                state.set_prop(f.name, cleaned)
            except InvalidStateError as e:
                errors[f.name] = e.error_list

        if errors:
            raise InvalidStateError(errors)

    def clean(self):
        """
        Hook for doing any extra model-wide validation after clean() has been
        called on every field by self.clean_fields. Any InvalidStateError raised
        by this method will not be associated with a particular field; it will
        have a special-case association with the field defined by NON_FIELD_ERRORS.
        """
        pass