from os.path import expanduser


class ProvisionerOptions(object):
    """
    Describes the connection and runtime options the
    provisioner uses.
    """
    __slots__ = ['_remote_user','_private_key','_sudo',
        '_sudo_user','_ignore_errors','_verbosity']

    def __init__(self, remote_user="root", private_key="~/.ssh/id_rsa",
        sudo=True, sudo_user="root", ignore_errors=False,
        verbosity=0):
        """
        Initialize a new :class:`ProvisionerOptions` instance.

        :param remote_user:
            connect as this user.
        :param private_key:
            use this file to authenticate the connection.
        :param sudo:
            run operations with sudo (nopasswd). Default: ``True``.
        :param sudo_user:
            desired sudo user. Default: ``'root'``.
        :param ignore_errors:
            specifies if errors are fatal. Default is ``False``.
        :param verbosity:
            specifies the verbosity of command-line output.
        """
        self._remote_user = remote_user
        self._private_key = private_key
        self._sudo = sudo
        self._sudo_user = sudo_user
        self._ignore_errors = ignore_errors
        self._verbosity = verbosity

    def as_dict(self):
        """
        Returns the provisioner configuration as a :class:`dict`.
        """
        return {
            'remote_user': self._remote_user,
            'private_key': self._private_key,
            'sudo': self._sudo,
            'sudo_user': self._sudo_user,
            'ignore_errors': self._ignore_errors,
            'verbosity': self._verbosity
        }

    def clone(self, **replacements):
        """
        Return a new :class:`ProvisionerOptions` configured with
        the specified `replacements`.
        """
        kwargs = self.as_dict()
        kwargs.update(replacements)
        return type(self)(**kwargs)

    def configure_runner(self, cls, *args, **kwargs):
        """
        Configure a runner `cls` with the options provided to the
        constructor. Usually `cls` is an :mod:`ansible.playbook.`Playbook`
        or a :mod:`ansible.runner.Runner` instance.

        :param cls:
            the class (instance of :class:`type`) to configure and
            instantiate.
        """
        kwargs.update(self.as_dict())
        kwargs['any_errors_fatal'] = not kwargs.pop('ignore_errors')
        return cls.fromoptions(*args, **kwargs)