from os.path import abspath, join, dirname
import sys
import copy
import os
import itertools

from sqlalchemy.schema import DDL
import jinja2

from libtng.template import TemplateDoesNotExist, get_template
from libtng import timezone
from libtng import const
from libtng import filefinder


SCHEMATA = [
    'auth',
    'hostcontroller',
    'cloudcontroller',
    'zone',
    'locale',
    'iso'
]
FILENAMES = ['types.sql', 'ddl.sql','history.sql','virtual.sql','procedures.sql','triggers.sql','initial_data.sql']
DEFAULT_PARAMS = {
    'schema_master'         : const.SCHEMA_MASTER,
    'schema_architecture'   : const.SCHEMA_ARCHITECTURE,
    'schema_domain'         : const.SCHEMA_DOMAIN,
    'schema_history'        : const.SCHEMA_HISTORY,
    'schema_public'         : const.SCHEMA_PUBLIC,
    'schema_pg'             : const.SCHEMA_PG_CATALOG,
    'schema_virtual'        : const.SCHEMA_VIRTUAL,
    'with_drop'             : False,
    'with_history_drop'     : False,
    'drop_cascade'          : False,
    'initdb'                : False
}


def find_components_sql(schemata=None):
    """
    Finds the SQL for all specified TNGEMS components.
    """
    schemata = schemata or SCHEMATA
    base_dir = 'templates/sql'
    sql = [join(base_dir, 'initdb.sql')]
    for schema, filename in itertools.product(schemata, FILENAMES):
        src = join(base_dir, schema, filename)
        sql.append(src)
    return sql


def get_template_instances():
    """
    Get a list of :class:`~tngems.template.Template` instances
    holding the schema SQL.
    """
    for x in find_components_sql():
        try:
            yield get_template(x)
        except TemplateDoesNotExist:
            continue


def aggregate_sql(**params):
    """
    Aggregates all SQL into a single string.
    """
    sql = []
    context_dict = copy.deepcopy(DEFAULT_PARAMS)
    context_dict.update(params)
    context_dict['generated'] = timezone.now()
    for t in get_template_instances():
        sql.append(t.render(context_dict))
    return 'BEGIN;\n' + '\n\n\n'.join(sql) + '\n\n\nCOMMIT;'


def render(**kwargs):
    kwargs.update({
        'with_drop'     : False,
        'drop_cascade'  : False,
        'with_data'     : True
    })
    return aggregate_sql(**kwargs)