import abc

from libtng import six
from libtng.cqrs.exceptions import MissingCommandHandler
from libtng.module_loading import import_string


class HandlersProvider(object):
    """
    Maps commands to their handlers.
    """

    def map(self, handlers):
        """Maps a list of command/handlers in the registry.

        The list can contain either tuples of types; or strings
        holding the fully-qualified name of a command and/or
        handler.
        """
        for Command, CommandHandler in handlers:
            if isinstance(Command, six.string_types):
                Command = import_string(Command)
            if isinstance(CommandHandler, six.string_types):
                CommandHandler = import_string(CommandHandler)
            self.register_handler(Command, CommandHandler)

    def register_handler(self, command_class, handler_class):
        """
        Registers `handler_class` as the handler for `command_class`.
        """
        self.__handlers[command_class] = handler_class

    def get(self, command_type):
        """Returns the handler for a command type."""
        return self.__handlers[command_type]

    def get_handler(self, gateway, runner, command, alias='default'):
        """
        Get a handler for the provided `command`.
        """
        try:
            return self.__handlers[type(command)](gateway, runner)
        except KeyError:
            raise MissingCommandHandler(
                "No handler registered for {0}".format(type(command).__name__))

    def __init__(self):
        self.__handlers = {}


registry = HandlersProvider()