


class Provisioner(object):
    """
    Provisioner configuration; comparable to a
    :class:`ansible.runner.Runner` object.
    """

    @classmethod
    def fromoptions(cls, options):
        kwargs = options.as_dict()
        kwargs['_options'] = options
        return cls(**kwargs)

    def __init__(self, remote_user='root', private_key='~/.ssh/id_rsa',
        ignore_errors=True, verbosity=0, _options=None, **kwargs):
        """
        Initialize a new provisioner.

        :param remote_user:
            connect to remote hosts as this user.
        :param private_key:
            a string pointing to a file on the local filesystem holding an
            RSA private key.
        :param ignore_errors:
            a boolean indicating if errors during provisioning should be
            ignored instead of raised.
        :param verbosity:
            specifies the verbosity of stdout.
        """
        self._remote_user = remote_user
        self._private_key = private_key
        self._ignore_errors = ignore_errors
        self._verbosity = verbosity
        self._options = _options

    def provision(self, provisionable, extra_params=None, **overrides):
        """
        Provision the provided `provisionable`. You can
        optionally override some or all connections settings
        by specifying keyword arguments. For more information
        refer to :meth:`libtng.provisioning.ProvisionerOptions.clone()`.
        """

        options = self._options.clone(**overrides)
        inventory = provisionable.get_provisioner_inventory()
        op = provisionable.get_provisioner_operation()
        return op.run(inventory, provisionable, options, extra_params)

    def provisionmany(self, provisionables, extra_params, **overrides):
        """
        Provision the provided `provisionables`, an iterable yielding
        :class:`~libtng.provisioning.Provisionable` instances. You can
        optionally override some or all connections settings
        by specifying keyword arguments. For more information
        refer to :meth:`libtng.provisioning.ProvisionerOptions.clone()`.
        """
        options = self._options.clone(**overrides)
        ops = collections.defaultdict(lambda: {'inv': Inventory()})
        for p in provisionables:
            op = p.get_provisioner_operation()
            ops[op]['inv'] &= p.get_provisioner_inventory()
            ops[op]['provisionable'] = p

        results = []
        for op, p in ops.items():
            op.run(p['inv'], p['provisionable'], options, extra_params)

    def get_single_node_inventory(self, host, groups=None, **hostvars):
        """
        Return a new :class:`Inventory` targeting a single host.

        :param host:
            specifies the hostname or IP-address.
        :param groups:
            specifies the groups `host` is a member of; may be ``None``.
        :param **hostvars:
            keyword argument providing a context for Ansible.
        """
        if isinstance(groups, six.string_types):
            groups = [groups]
        hosts = [host] if isinstance(host, six.string_types)\
            else host
        groups = groups or ['all']
        inventory = Inventory()
        for host, group in itertools.product(hosts, groups):
            inventory.add_host(group, host)
            inventory.set_hostvars(host, hostvars)
        return inventory

    def run_playbook(self, inventory, playbook):
        """
        Run an Ansible playbook.

        :param playbook:
            a string pointing a file on the local filesystem,
            identifying the playbook.
        :param inventory:
            a :class:`Inventory` instance.
        """
        from os.path import join
        from libtng import environ
        from _ansible import PlayBook
        base_dir = environ.getenv('ANSIBLE_PLAYBOOK_BASEDIR')
        src = join(base_dir, playbook)
        pb = PlayBook.fromcommand(self, inventory, src, base_dir=base_dir,
            verbosity=self._verbosity)
        return pb.run()

    def run_module(self, inventory, group, module_name, **kwargs):
        """
        Run an Ansible module on the specified inventory.

        :param inventory:
            a :class:`Inventory` instance.
        :param group:
            the group in the inventory to run the module on.
        :param kwargs:
            module parameters.
        :returns:
            a list of :class:`~libtng.provisioning.TaskResult`
            instances.
        """
        from _ansible import Runner
        from libtng import environ
        base_dir = environ.getenv('ANSIBLE_PLAYBOOK_BASEDIR')
        sudo_user = kwargs.pop('sudo_user', 'root')
        remote_user = kwargs.pop('remote_user', 'root')
        sudo = kwargs.pop('sudo', True)
        runner = Runner.fromprovisioner(self, inventory, module_name,
            args_to_string(**kwargs), remote_user=remote_user or self._remote_user,
            sudo_user=sudo_user, base_dir=base_dir, verbosity=self._verbosity,
            sudo=sudo)
        return runner.run()

    def get_playbook(self, playbook_cls, playbook_src, inventory, **kwargs):
        return playbook_cls(
            playbook=playbook_src,
            inventory=inventory,
            remote_user=self._remote_user,
            private_key_file=self._private_key,
            any_errors_fatal=not self._ignore_errors,
            **kwargs
        )

    def get_runner(self, runner_cls, **kwargs):
        return runner_cls(
            private_key_file=self._private_key,
            **kwargs
        )
