"""
Command-Query Responsibility Segregation framework (CQRS). Separates
destructive operations from non-destructive (read) operations; CQRS
is "the creation of two objects where there was previously one.
The separation occurs based upon whether the methods are a command
or a query." (Greg Young).

Example commands:
- Update the state of a relation in a relational database.
- Create a directory on a filesystem.

Example queries:
- Get a result set for a query on a relational database.
- Read the contents of a directory on a filesystem.
"""
import functools

from libtng.cqrs.command import Command
from libtng.cqrs.command import CommandAttribute
from libtng.cqrs.command import CommandParameter
from libtng.cqrs.command import CommandValidationError
from libtng.cqrs.command import CommandInspector
from libtng.cqrs.command import CommandMetadata
from libtng.cqrs.command import CommandRequest
from libtng.cqrs.command import CommandRequest as GatewayRequest
from libtng.cqrs.gateway import SimpleGateway
from libtng.cqrs.gateway import IGateway
from libtng.cqrs.gateway import GatewayResponse
from libtng.cqrs.gateway import Transaction
from libtng.cqrs.runner import Runner
from libtng.cqrs.runner import CommandExecutionResult
from libtng.cqrs.exceptions import MissingCommandHandler
from libtng.cqrs.exceptions import UnprocessableCommandException
from libtng.cqrs.exceptions import CommandDeserializationError
from libtng.cqrs.exceptions import InvalidCommandParameter
from libtng.cqrs.exceptions import InvalidCommandParameters
from libtng.cqrs.exceptions import GatewayNotResponding
from libtng.cqrs.exceptions import GatewayRejectedCommand
from libtng.cqrs.exceptions import DuplicateCommand
from libtng.cqrs.exceptions import UnknownCommand
from libtng.cqrs.exceptions import InvalidRequestData
from libtng.cqrs.handler import CommandHandler
from libtng.cqrs.provider import HandlersProvider
from libtng.cqrs.provider import registry


__all__ = [
    'Command',
    'CommandHandler',
    'Runner',
    'SimpleGateway',
    'HandlersProvider',
    'Transaction',
    'MissingCommandHandler',
    'inject_state',
    'registry'
]


def handles(command_cls, provider=None):
    """
    Decorator to be used on :class:`~libtng.cqrs.CommandHandler`
    subclasses in order to specify the command they handle.

    :param command_cls:
        a :class:`~libtng.cqrs.Command` subclass.
    :param provider:
        specifies the registry where the mapping of the command
        to it's handler will be stored.
    """
    global registry
    provider = provider or registry
    def wrapper(cls):
        provider.register_handler(command_cls, cls)
        return cls

    return wrapper
