import abc
import collections
import uuid

from libtng.datastructures import ImmutableDict
from libtng import timezone
from libtng import six
from libtng.cqrs.provider import registry
from libtng.cqrs.command.inspector import CommandInspector
from libtng.cqrs.command.metadata import CommandMetadata
from libtng.cqrs.command.meta import CommandMeta
from libtng.cqrs.exceptions import InvalidCommandParameters
from libtng.cqrs.exceptions import InvalidCommandParameter
from libtng.cqrs.const import PENDING
import libtng.io

__all__ = ['Command']


isoformat = lambda x: datetime.datetime.strptime("%Y-%m-%d %H:%M:%S.%f")
datetime_from_string = lambda x: isoformat(x) if (x is not None) else None


class Command(six.with_metaclass(CommandMeta), collections.Mapping,
    libtng.io.Serializable):
    """
    Describes a destructive command on a certain part of a system.
    """

    @property
    def metadata(self):
        return self._metadata

    @property
    def inspector(self):
        return self._inspector

    def __init__(self, *args, **kwargs):
        self._inspector = CommandInspector()
        params = {}
        errors = []
        for param in self._inspector.get_parameters(self):
            value = kwargs.get(param.attname)
            try:
                value = param.validate(value)
                params[param.attname] = value
            except InvalidCommandParameter as e:
                errors.append(e)
        if errors:
            raise InvalidCommandParameters(parameters=errors)
        self._set_parameters(params)
        self._metadata = CommandMetadata(self)
        self._result = None

    @classmethod
    def handler(cls, handler_class):
        """
        Class-decorator to use on a :class:`~libtng.cqrs.CommandHandler`
        implementation to register it for `cls`.
        """
        handler_class.command_type = cls
        cls.handler = handler_class
        return handler_class

    def as_dict(self):
        """Project the command as a dictionary."""
        return self.inspector.as_dict(self)

    def as_tuple(self):
        """Project the command as a named tuple."""
        inspector = self.inspector
        return self.tuple_type(**inspector.as_dict(self))

    def _set_parameters(self, params):
        self._parameters = {}
        for param in self._inspector.get_parameters(self):
            self._parameters[param.attname] = params.pop(param.attname)
        self._parameters = ImmutableDict(self._parameters)

    def set_result(self, result):
        """Set the result of command processing to the command.

        Args:
            result (libtng.cqrs.CommandExecutionResult): the result.

        Returns:
            None
        """
        self._result = result

    def __getitem__(self, key):
        if key not in self._attributes:
            raise KeyError(key)
        return getattr(self, key)

    def __iter__(self):
        return self.as_dict().__iter__()

    def __len__(self):
        return len(self.as_dict().keys())

    def __hash__(self):
        return self.inspector.hash(self)

    def __eq__(self, other):
        return hash(self) == hash(other)
