import json

from libtng.spec import Specification
from libtng.module_loading import import_string
from libtng.encoding import force_text
from libtng.encoding import force_bytes
from libtng.cqrs.exceptions import CommandDeserializationError
from libtng.cqrs.exceptions import UnknownCommand
from libtng.cqrs.exceptions import InvalidRequestData


class CommandRequest(object):
    """Represents the request to the gateway to execute a
    state-modifying command.

    Commands are composed by two components:

    -   Metadata describing the command type, date/time of invocation,
        and arbitrary user-defined properties.
    -   The command parameters.
    """
    __slots__ = ['_metadata','_params']
    specification = (
        Specification(lambda request: isinstance(request.user_id, int)) &
        Specification(lambda request: isinstance(request.client_id, int)) &
        Specification(lambda request: isinstance(request.command_name, str))
    
    )

    @property
    def user_id(self):
        return self._metadata.get('user_id')

    @property
    def client_id(self):
        return self._metadata.get('client_id')

    @property
    def command_name(self):
        return self._metadata.get('type')

    @classmethod
    def fromrawdata(cls, raw_data, deserializer=None):
        """Initialize a new :class:`CommandRequest` from serialized data
        received through the wire.

        Args:
            raw_data (str): the serialized data.
            deserializer: a callable that deserializes the data into the
                command components. If `deserializer` is ``None``, it
                defaults to JSON.

        Returns:
            CommandRequest

        Raises:
            CommandDeserializationError: the command could not be deserialized.
        """
        deserializer = deserializer or (lambda x: json.loads(force_text(x)))
        if hasattr(deserializer, 'deserialize'):
            deserializer = deserializer.deserialize
        try:
            metadata, params, *extra = deserializer(raw_data)
        except ValueError:
            raise CommandDeserializationError
        return cls(metadata, params, *extra)

    @classmethod
    def deserialize(cls, raw_data, serializer):
        return cls.fromrawdata(raw_data, deserializer=serializer)

    @property   
    def command(self):
        return self.command_type(**self._params)

    @property
    def command_type(self):
        return import_string(self._metadata['type'])

    def __init__(self, metadata, params, *extra):
        """Initiailize a new :class:`CommandRequest` instance.

        Args:
            metadata (dict): metadata describing the command.
            params (dic): the command parameters.
        """
        self._metadata = metadata
        self._params = params

    def validate(self, provider=None):
        """Validates the request to put a command.

        Args:
            provider (libtng.cqrs.HandlersProvider): optionally check if a
                handler has been registered for the command.

        Returns:
            None
        """
        if not self.specification.is_satisfied(self):
            raise InvalidRequestData
        try:
            command_type = self._metadata['type']
            Command = import_string(command_type)
            if provider is not None and \
            not provider.is_known_command_type(Command):
                raise UnknownCommand
        except (KeyError, ImportError):
            raise UnknownCommand
        command = Command(**self._params)


    def handle(self, provider):
        """Handles the command specified by the request."""
        handler = provider.get(self.command_type)
        return handler.handle(self.command)

    def serialize(self, serializer, *args, **kwargs):
        """Serializes the :class:`CommandRequest`.

        Args:
            serializer: a callable implementing the 
                :class:`~libtng.io.Serializer` interface.

        Returns:
            bytes
        """
        return force_bytes(serializer.serialize(
            list(self)))

    def __iter__(self):
        return iter([self._metadata, self._params])

    def __str__(self):
        try:
            return "<Command: {0}>".format(
                self._metadata['type'])
        except KeyError:
            return "<Command: Invalid data received>"


