import abc
import collections
import threading
import random
import sys

from libtng import six
from libtng import timezone
from libtng.cqrs.command.request import CommandRequest as GatewayRequest
from libtng.cqrs.command.metadata import CommandMetadata
from libtng.cqrs.runner import Runner
from libtng.cqrs.gateway.transaction import Transaction
from libtng.cqrs.exceptions import DuplicateCommand


__all__ = [
    'CommandExecution'
]


class IGateway(six.with_metaclass(abc.ABCMeta)):

    def __init__(self, runner=None, transaction_factory=None):
        self._runner = runner or Runner()
        self._unique_commands = collections.defaultdict(set)
        self._transaction_factory = None

    @abc.abstractmethod
    def dispatch(self, command, *args, **kwargs):
        """
        Executes the given command.
        """
        pass

    def get_metadata(self, command, **extra):
        """Get the metadata for a command."""
        metadata = CommandMetadata(command)
        metadata.update(extra)
        return metadata

    def put(self, command, **extra_metadata):
        """Put a command an assess if it can be executed; proceed to
        invoke the command-runner to execute. Returns a :class:`libtng.
        cqrs.Transaction` instance representing the transaction.

        Args:
            command (libtng.cqrs.Command): a concrete command implementation.
            **extra_metadata: extra metadata parameters.

        Returns:
            libtng.cqrs.Transaction
        """
        metadata = self.get_metadata(command, **extra_metadata)
        response = self.dispatch(command, metadata)
        return response

    def get_request(self, command, **extra_metadata):
        """Get a new request.
        
        Args:
            command (libtng.cqrs.Command): a concrete command implementation.
            **extra_metadata: extra metadata parameters.

        Returns:
            libtng.cqrs.GatewayRequest
        """
        metadata = self.get_metadata(command, **extra_metadata)
        return GatewayRequest(metadata.as_dict(), command.as_dict())


    @abc.abstractmethod
    def dispatch(self, command, metadata, *extra):
        """Dispatches the command to the command-processing gateway
        backend.

        Args:
            command: the command to dispatch.
            metadata: metadata describing the command.

        Returns:
            libtng.cqrs.Transaction
        """
        raise NotImplementedError
