import abc

from libtng import six


class CommandHandler(six.with_metaclass(abc.ABCMeta)):
    """
    Base class for command handlers. Subclasses must override
    the :meth:`CommandHandler.handle` method.

    Subclasses may specify a ``rollback()`` method, which reverts
    **all** changes made by :meth:`CommandHandler.handle()`. It is
    assumed that if a subclass does not specify this method, the
    handler is not able to rollback.

    The ``rollback()`` method receives the command and any arguments
    passed to the gateway.
    """
    
    @abc.abstractmethod
    def handle(self, command, *args, **kwargs):
        """
        Execute the command. Receives the command and additional
        parameters passed to the gateway.
        """
        pass

    def run_command(self, command, func, *args, **kwargs):
        # Private method to allow setting up a specific command
        # execution context.
        return func(command, *args, **kwargs)
