#!/usr/bin/env python3
# -*- coding: utf-8 -*-

#
# Copyright © 2013 Felix Crux <felixc@felixcrux.com> and the Authors.
# Released under the terms of the MIT License (Expat Version).
# See the LICENSE and AUTHORS files for further details.
#


import datetime
import os
import tempfile
import unittest

import libwedger


class TestGetTransactionRegister(unittest.TestCase):

    def setUp(self):
        self.ledger_file = tempfile.NamedTemporaryFile("w+")
        os.environ["LEDGER_FILE"] = self.ledger_file.name
        os.environ["HOME"] = tempfile.gettempdir()

    def tearDown(self):
        self.ledger_file.close()

    def test_no_transactions(self):
        self.assertEqual(libwedger.get_transaction_register(), [])

    def test_one_transaction(self):
        self.ledger_file.write("2013-11-12 Writing Tests\n"
                               "  Time  -30 minutes\n"
                               "  QA")
        self.ledger_file.flush()
        self.assertEqual(
            libwedger.get_transaction_register(),
            [{"date": datetime.date(2013, 11, 12),
              "payee": "Writing Tests",
              "accounts": [{"account": "Time", "amount": "-30 minutes"},
                           {"account": "QA", "amount": "30 minutes"}]}])

    def test_multiple_transactions(self):
        self.ledger_file.write("2013-11-12 Writing Tests\n"
                               "  Time  -30 minutes\n"
                               "  QA\n"
                               "2013-11-11 Planning\n"
                               "  Time  -15 minutes\n"
                               "  Plans\n")
        self.ledger_file.flush()
        self.assertEqual(
            libwedger.get_transaction_register(),
            [{"date": datetime.date(2013, 11, 12),
              "payee": "Writing Tests",
              "accounts": [{"account": "Time", "amount": "-30 minutes"},
                           {"account": "QA", "amount": "30 minutes"}]},
             {"date": datetime.date(2013, 11, 11),
              "payee": "Planning",
              "accounts": [{"account": "Time", "amount": "-15 minutes"},
                           {"account": "Plans", "amount": "15 minutes"}]}])

    def test_multiple_entries_per_transaction(self):
        self.ledger_file.write("2013-11-12 Writing Tests\n"
                               "  Time      -30 minutes\n"
                               "  QA        15 minutes\n"
                               "  Plans     5 minutes\n"
                               "  Bugfixes")
        self.ledger_file.flush()
        self.assertEqual(
            libwedger.get_transaction_register(),
            [{"date": datetime.date(2013, 11, 12),
              "payee": "Writing Tests",
              "accounts": [{"account": "Time", "amount": "-30 minutes"},
                           {"account": "QA", "amount": "15 minutes"},
                           {"account": "Plans", "amount": "5 minutes"},
                           {"account": "Bugfixes", "amount": "10 minutes"}]}])

    def test_commodities_with_price_transaction(self):
        self.ledger_file.write("2013-11-12 Writing Tests\n"
                               "  Time  -30 minutes @ $1\n"
                               "  QA")
        self.ledger_file.flush()
        self.assertEqual(
            libwedger.get_transaction_register(),
            [{"date": datetime.date(2013, 11, 12),
              "payee": "Writing Tests",
              "accounts": [{"account": "Time", "amount": "-30 minutes"},
                           {"account": "QA", "amount": "$30"}]}])
