from linfir.mdtex import abbrev
from linfir.mdtex.resources import resource
import re
import roman
import string

__all__ = ['HTML', 'LaTeX', 'MD']


def tag(x):
    assert isinstance(x, tuple) and len(x) > 0 and isinstance(x[0], str)
    return x[0]


class Base:
    def render_inline(self, x):
        assert isinstance(x, list)
        return ''.join([self.render_inline_1(y) for y in x])

    def render_inline_1(self, x):
        t = tag(x)

        if t == 'text':
            return self.text(x[1])
        if t == 'emph':
            return self.emph(self.render_inline(x[1]))
        if t == 'math':
            return self.math(x[1])
        if t == 'mathfull':
            return self.mathfull(x[1])
        if t == 'abbrev':
            return self.abbrev(x[1])
        if t == 'comment':
            return self.comment(x[1])

        raise ValueError("Unknown inline tag {!r}".format(t))

    def render_block(self, x):
        assert isinstance(x, list)
        return '\n\n'.join([self.render_block_1(y) for y in x]) + '\n'

    def render_block_1(self, x):
        t = tag(x)
        if t == 'Header':
            return self.header(x[1], self.render_inline(x[2]))
        if t == 'Theorem':
            return self.theorem(self.render_inline(x[1]))
        if t == 'Paragraph':
            return self.paragraph(self.render_inline(x[1]))
        if t in ('Itemize', 'Enumerate'):
            return self.render_list(x)
        raise ValueError("Unknown block tag {!r}".format(t))

    def render(self, x):
        x = self.doc(x).strip()
        x = re.sub(r" +\n", "\n", x.strip())
        x = re.sub(r"\n\n\n+", "\n\n", x)
        return x + "\n"


class HTML(Base):
    def text(self, x):
        x = x.replace(";", "&nbsp;;")
        x = x.replace(":", "&nbsp;:")
        x = x.replace("'", "&rsquo;")
        return x

    def emph(self, x):
        return "<em>" + x + "</em>"

    def math(self, x):
        return "\\(" + x + "\\)"

    def mathfull(self, x):
        return "$$ " + x + " $$"

    def abbrev(self, x):
        return "<code>\\" + x + "</code>"

    def comment(self, x):
        return ""

    def header(self, n, x):
        return "<h{0}>{1}</h{0}>".format(n, x)

    def theorem(self, x):
        return "<h5>{}</h5>".format(x)

    def paragraph(self, x):
        return "<p>{}</p>".format(x)

    def render_list(self, x):
        t = tag(x)
        h = {'Enumerate': 'ol', 'Itemize': 'ul'}
        assert t in h
        L = ["<" + h[t] + ">"]
        for i in x[1]:
            L.append("<li>" + self.render_item(i) + "</li>")
        L.append("</" + h[t] + ">")
        return '\n'.join(L)

    def render_item(self, x):
        assert tag(x) == 'Item'
        br = False
        out = ""
        for y in x[1]:
            t = tag(y)
            if t == 'Paragraph':
                if br:
                    out += "<br/>\n"
                out += self.render_inline(y[1])
            elif t in ('Itemize', 'Enumerate'):
                out += self.render_block_1(y)
            else:
                raise ValueError("Unknown sub block tag {!r}".format(t))
            br = (t == 'Paragraph')
        return out.strip()

    def doc(self, x):
        return self.render_block(x['doc'])

    def full_doc(self, x, pdf):
        m = x['meta']
        t = string.Template(resource("template.html"))
        p = resource("macros.tex")
        p += resource("macros_html.tex")
        p += m.get('preamble', "")
        return t.substitute(
            body=self.doc(abbrev.expand(x)),
            preamble=abbrev.expand_symbols(p),
            imprimable=pdf,
            title=m.get('title', "UNTITLED"),
            css=resource("style.css"))


class LaTeX(Base):
    def __init__(self):
        self.thm_nb = 0

    def text(self, x):
        return x

    def emph(self, x):
        return "\\emph{" + x + "}"

    def math(self, x):
        return "$" + x + "$"

    def mathfull(self, x):
        return "\\[" + x + "\\]"

    def abbrev(self, x):
        return "\\verb|\\" + x + "|"

    def comment(self, x):
        return ""

    def header(self, n, x):
        return "\\" + (n-1) * "sub" + "section{" + x + "}"

    def theorem(self, x):
        self.thm_nb += 1
        x = x.strip()
        m = re.match("(.*?) (\(.*\))$", x)
        if m:
            x = "{} {} {}".format(m.group(1), self.thm_nb, m.group(2))
        else:
            x += " {}".format(self.thm_nb)
        return "\\paragraph{" + x + "}"

    def paragraph(self, x):
        return x + "\n"

    def render_list(self, x):
        t = tag(x).lower()
        assert t in ('enumerate', 'itemize')
        L = ["\\begin{" + t + "}"]
        for i in x[1]:
            L.append("\\item " + self.render_item(i))
        L.append("\\end{" + t + "}")
        return '\n'.join(L)

    def render_item(self, x):
        assert tag(x) == 'Item'
        x = x[1]
        return self.render_block(x).strip()

    def doc(self, x):
        return self.render_block(x['doc'])

    def full_doc(self, x):
        m = x['meta']
        t = string.Template(resource("template.tex"))
        p = resource("macros.tex")
        p += resource("macros_latex.tex")
        p += m.get('preamble', "")
        return t.substitute(
            body=self.doc(abbrev.expand(x)),
            preamble=abbrev.expand_symbols(p),
            title=m.get('title', "UNTITLED"))


class MD(Base):
    def __init__(self):
        self.indent = 0
        self.list_level = 0
        self.thm = 0

    def text(self, x):
        return x.replace("\n", "\n" + self.indent * " ")

    def emph(self, x):
        return "*" + x + "*"

    def math(self, x):
        return "$" + x + "$"

    def mathfull(self, x):
        return "$$ " + x + " $$"

    def abbrev(self, x):
        return "\\" + x

    def comment(self, x):
        return "<!--" + x + "-->"

    def header(self, n, x):
        return n * "#" + " " + x

    def theorem(self, x):
        self.thm += 1
        return "!!{}!! ".format(self.thm) + x

    def paragraph(self, x):
        return x + "\n"

    def doc(self, x):
        return md_meta(x['meta']) + self.render_block(x['doc'])

    def render_list(self, x):
        self.list_level += 1
        out = ""
        t = tag(x)
        items = x[1]
        assert isinstance(items, list)

        delta, B = md_bullets(t, self.list_level, len(items))
        first = True
        indent = self.indent
        self.indent += delta
        for b, y in zip(B, items):
            assert tag(y) == 'Item'
            if not first:
                out += " " * indent
            first = False
            out += b + self.render_items(y[1])

        self.indent -= delta
        self.list_level -= 1
        return out.rstrip()

    def render_items(self, L):
        out = ""

        assert isinstance(L, list)
        skip = False
        first = True

        for x in L:
            t = tag(x)

            if not first:
                out += self.indent * " "
            first = False

            if t == 'Paragraph':
                if skip:
                    out += "\n" + self.indent * " "
                out += self.render_inline(x[1]) + "\n"
                skip = True
            elif t in ['Enumerate', 'Itemize']:
                out += self.render_list(x) + "\n"
            else:
                raise ValueError("Unknown sub tag {!r}".format(t))

        return out


def md_bullets(t, level, nb):
    B = []
    for n in range(nb):
        if t == 'Itemize':
            if level % 3 == 1:
                B.append("-")
            elif level % 3 == 2:
                B.append("*")
            else:
                B.append("+")
        elif t == 'Enumerate':
            if level % 3 == 1:
                B.append("{}.".format(n+1))
            elif level % 3 == 2:
                assert n <= 26, "Too many items!"
                B.append("{}.".format(chr(ord('a')+n)))
            else:
                B.append("{}.".format(roman.toRoman(n+1).lower()))
        else:
            raise ValueError("Unknown list tag {!r}".format(t))

    delta = max([len(b) for b in B]) + 1
    delta += (4 - delta) % 4
    B = [b + " " * (delta - len(b)) for b in B]

    return delta, B


def md_meta(m):
    if not m:
        return ""
    out = "---\n"
    out2 = ""
    for k in sorted(m.keys()):
        if not m[k]:
            continue
        if k == 'preamble':
            out2 += m[k].strip()
        else:
            out += "{}: {}\n".format(k, m[k])
    out += "---\n\n" + out2
    return out.rstrip() + "\n\n"
