# author   : Johann-Mattis List
# email    : mattis.list@uni-marburg.de
# created  : 2013-07-17 10:40
# modified : 2013-11-23 09:20
"""
Module handels all global parameters used in a LingPy session.
"""

__author__="Johann-Mattis List"
__date__="2013-11-23"

# builtin imports
from datetime import datetime,date
import os

# internal imports
from ._settings import rcParams
from .data.model import Model, load_dvt

# load diacritics, vowels, tones
diacritics, vowels, tones = load_dvt()

# these are lexstat-specific parameters, all prefixed by "lexstat"
lexstat = dict(
        lexstat_transform      = {
                    'A':'C',
                    'B':'C',
                    'C':'C',
                    'L':'c',
                    'M':'c',
                    'N':'c',
                    'X':'V', #
                    'Y':'V', #
                    'Z':'V', #
                    'T':'T', #
                    '_':'_'
                    },
        lexstat_runs           = 1000,
        lexstat_modes          = [("global",-2,0.5),("local",-1,0.5)],
        lexstat_rands          = 1000,
        lexstat_limit          = 10000,
        lexstat_scoring_method = 'shuffle',
        lexstat_ratio          = (2,1),
        lexstat_vscale         = 0.5,
        lexstat_threshold      = 0.7,
        lexstat_cluster_method = 'upgma',
        lexstat_preprocessing_method = 'sca',
        lexstat_preprocessing_threshold = 0.7
        )
rcParams.update(lexstat)

# these are alignment-specific parameters, all prefixed by "align"
alignments = dict(
        align_mode                                    = 'global',
        align_modes                                   = [
                ('global',-2,0.5),
                ('local',-1,0.5),
                ],
        align_scale                                   = 0.5,
        align_factor                                  = 0.3,
        align_gap_weight                              = 0.5,
        align_classes                                 = True,
        align_sonar                                   = True,
        align_scorer                                  = {},
        align_tree_calc                               = 'neighbor',
        align_gop                                     = -2,
        align_transform                               = {

            # new values for alternative prostrings
            'A' : 1.6,  # initial
            'B' : 1.3, # syllable-initial
            'C' : 1.2,  # ascending
            'L' : 1.1,  # descending
            'M' : 1.1,  # syllable-descending
            'N' : 0.5,  # final
            'X' : 3.0,  # vowel in initial syllable
            'Y' : 3.0,  # vowel in non-final syllable
            'Z' : 0.7,  # vowel in final syllable
            'T' : 1.0,  # Tone
            '_' : 0.0   # break character
            },
        align_notransform                             = {
            # new values for alternative prostrings
            'A' : 1,  # initial
            'B' : 1, # syllable-initial
            'C' : 1,  # ascending
            'L' : 1,  # descending
            'M' : 1,  # syllable-descending
            'N' : 1,  # final
            'X' : 1,  # vowel in initial syllable
            'Y' : 1,  # vowel in non-final syllable
            'Z' : 1,  # vowel in final syllable
            'T' : 1,  # Tone
            '_' : 1   # break character
            },
        align_stamp                                   = """# MSA
# dataset    : {0}
# collection : {1} 
# aligned by : LingPy-2.2 <www.lingpy.org>
# created on : {2}
# parameters : {3}
"""

        )
rcParams.update(alignments)

# dictionary stores basic parameters that are used during a LingPy session
rcParamsUpd = dict(
        verbose                    = False,
        debug                      = False,
        schema                     = 'qlc',
        asjp                       = Model('asjp'),
        sca                        = Model('sca'),
        dolgo                      = Model('dolgo'),
        _color                     = Model('color'),
        art                        = Model('art'),
        jaeger                     = Model('jaeger'),
        diacritics                 = diacritics,
        model                      = Model('sca'),
        vowels                     = vowels,
        tones                      = tones,
        figsize                    = (10,10),
        combiners                  = '\u0361\u035c',
        breaks                     = '.-',
        stress                     = "ˈˌ'",
        merge_vowels               = True,
        unique_sequences           = True,
        comment                    = '#',
        restricted_chars           = '_T',
        scale                      = 0.5,
        factor                     = 0.3,
        gap_weight                 = 0.5,
        classes                    = True,
        sonar                      = True,
        scorer                     = {},
        tree_calc                  = 'neighbor',
        gop                        = -2,
        ref                        = 'cogid',
        identical_scorer_warning   = "[i] An identical scoring function has already been calculated, force recalculation by setting 'force' to 'True'.",
        overwrite_scoring_function = "[i] A different scoring function has already been calculated, overwriting previous settings.",
        empty_consensus_warning    = '[WARNING] There are empty segments in the consensus!',
        sonority_consensus_warning = '[WARNING] Sonority profile consensus could not be calculated!',
        deprecation_warning        = "[WARNING] Use of '{0}' is deprecated, use '{1}' instead.",
        missing_module             = "[WARNING] Module '{0}' could not be loaded. Some methods may not work properly.",
        test_path                  = os.path.join(rcParams['_path'], 'tests', 'test_data', '')
        )
rcParams.update(rcParamsUpd)

# define parameters for specific semi-verbose output
rcParams.update(
        dict(
            _sverb = True,
            _sverb_fchar = '-',
            _sverb_tbar_len = 100,
            _sverb_tchar = '+',
            )
        )

# define aliases for parameters
kw_base = dict(
    filename = ('filename', 'fn'),
    M_file_written = ('M_fw',),
    merge_vowels = ('mv',),
    sca = ("model",),
    _sverb = ("progressbar", "semi_verbose",)
    )
alias = {}
for key in kw_base:
    # set key as key, just to make sure that the keyword always occurs in the
    # alias dictionary
    alias[key] = key

    # set all the alias values
    for value in kw_base[key]:
        alias[value] = key

# function changes parameters
def rc(rval=None, **keywords):
    """
    Function changes parameters globally set for LingPy sessions.

    Parameters
    ----------
    rval : string (default=None)
        Use this keyword to specify a return-value for the rc-function. 
    schema : {"ipa", "asjp"}
        Change the basic schema for sequence comparison. When switching to
        "asjp", this means that sequences will be treated as sequences in ASJP
        code, otherwise, they will be treated as sequences written in basic
        IPA.
    verbose : bool (default=False)
        Use this keyword in order to switch to verbose output. This will be
        useful when using complex methods, in order to understand what the
        program is actually doing.
    debug : bool (default=False)
        Use this keyword to switch to debug-mode. It will give specific,
        internal output that is much more technical than the output resulting
        from "verbose".

    Notes
    -----
    This function is the standard way to communicate with the *rcParams*
    dictionary which is not imported as a default. If you want to see which
    parameters there are, you can load the rcParams dictonary directly::

    >>> from lingpy.settings import rcParams

    However, be careful when changing the values. They might produce some
    unexpected behavior.

    Examples
    --------
    Import LingPy:

    >>> from lingpy import *
    
    Have the rc-function return a specific value, such as the path to the
    test-directory of LingPy:

    >>> rc("test_path")
    '/usr/lib/python3.3/site-packages/lingpyd/tests/test_data/'

    Change basic values. Switch to verbose output, for example:

    >>> rc(verbose=True)
    [i] Successfully changed parameters.
    
    """
    if rval:
        return rcParams[rval]
    
    for key in keywords:
        if key == "schema":
            if keywords[key] in ["qlc",'ipa']:
                diacritics,vowels,tones = load_dvt(path='')
                rcParams['asjp'] = Model('asjp')
                rcParams['sca'] = Model('sca')
                rcParams['dolgo'] = Model('dolgo')
                rcParams['art'] = Model('art')
                rcParams['diacritics'] = diacritics
                rcParams['vowels'] = vowels
                rcParams['tones'] = tones
                rcParams['_color'] = Model('color')
                rcParams['combiners']    = '\u0361\u035c'
                rcParams['breaks']       = '.-'
                rcParams['stress']       = "ˈˌ'"
                rcParams['merge_vowels'] = True
                rcParams['basic_orthography'] = 'fuzzy'
            elif keywords[key] in ['evolaemp','el','asjp']:
                diacritics,vowels,tones = load_dvt(path='el')
                rcParams['asjp'] = Model('asjp_el')
                rcParams['sca'] = Model('sca_el')
                rcParams['dolgo'] = Model('dolgo_el')
                rcParams['art'] = Model('art_el')
                rcParams['jaeger'] = Model('jaeger_el')
                rcParams['diacritics'] = diacritics
                rcParams['vowels'] = vowels
                rcParams['tones'] = tones
                rcParams['_color'] = Model('color_el')
                rcParams['combiners']    = '\u0361\u035c'
                rcParams['breaks']       = '.-'
                rcParams['stress']       = "ˈˌ'"
                rcParams['merge_vowels'] = False
                rcParams['basic_orthography'] = 'asjp'

        if key in alias:
            rcParams[alias[key]] = keywords[key]
        else:
            rcParams[key] = keywords[key]
    if rcParams['verbose']:
        print("[i] Successfully changed parameters.")


