# Copyright (c) 2012 Leif Johnson <leif@leifjohnson.net>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.


def defaults(words, tags, context):
    '''Extract features from a sequence of words and a sequence of tags.

    Given an array of words to be tagged and a (possibly shorter) array of
    hypothesized tags so far, extract a set of feature counts for the rightmost
    tag in the tag array. Returns an object mapping each feature to 1.

    This is a basic implementation that yields n-gram tag features like

      /t_{i-1} /t_i
      /t_{i-2} /t_{i-1} /t_i
      ...
      /t_{i-n} ... /t_{i-1} /t_i

    n-gram word features with lookahead like

      w_i/t_i
      w_{i-1} w_i/t_i w_{i+1}
      ...
      w_{i-n} ... w_i/t_i ... w_{i+n}

    and tagged n-gram word features with word lookahead like

      w_i/t_i
      w_{i-1}/t_{i-1} w_i/t_i w_{i+1}
      ...
      w_{i-n}/t_{i-n} ... w_i/t_i ... w_{i+n}

    Other feature extraction functions are certainly possible ; just define a
    function that accepts these arguments and returns an object mapping strings to
    numbers.
    '''
    c = len(tags) - 1

    t = '/%s' % tags[c]
    ts = [t]

    w = words[c]
    ws = ['%s%s' % (w, t)]

    wts = [ws[0]]

    yield ws[0]

    for i in range(1, context + 1):
        t = '/%s' % tags[c - i]
        ts = [t] + ts
        yield ' '.join(ts)

        w = words[c - i]
        ws = [w] + ws + [words[c + i]]
        yield ' '.join(ws)

        wts = ['%s%s' % (w, t)] + wts + [words[c + i]]
        yield ' '.join(wts)


class Counter(dict):
    '''A dictionary specialized for counting feature strings.'''

    def count(self, keys):
        for key in keys:
            self[key] = self.get(key, 0) + 1

    def __add__(self, items):
        if hasattr(items, 'iteritems'):
            items = items.iteritems()
        counts = Counter(self)
        for key, value in items:
            counts[key] = counts.get(key, 0) + value
        return counts

    def __iadd__(self, items):
        if hasattr(items, 'iteritems'):
            items = items.iteritems()
        for key, value in items:
            self[key] = self.get(key, 0) + value
        return self

    def __sub__(self, other):
        for key in set(self) | set(other):
            delta = self.get(key, 0) - other.get(key, 0)
            if delta:
                yield key, delta
