#!/usr/bin/env python

import os
import sys
import imp
import time
from datetime import datetime
from random import random
from copy import deepcopy
import inspect
import traceback
import multiprocessing
import Queue
import threading


class Scheduler(multiprocessing.Process):
    def __init__(self, queue, start_time, name, script, schedule, pacing, configuration={}):
        multiprocessing.Process.__init__(self)
        self.q = queue
        self.start_time = start_time
        self.name = name
        self.schedule = schedule
        self.pacing = pacing
        self.configuration = configuration
        # load Class from File
        mod = imp.load_source('module', script['File'])
        self.test = getattr(mod, script['Class'])

    def run(self):
        # here we do the Scheduling
        # "Schedule": {"Delay": 0, "Users": 2, "Rampup": 1},
        thread_refs = []
        # Delay
        if self.schedule['Delay'] > 0:
            time.sleep(self.schedule['Delay'])
        # Users
        if self.schedule['Users'] > 1 and 'Rampup' in self.schedule:
            # Rampup
            spacing = float(self.schedule['Rampup']) / (float(self.schedule['Users'])-1)
        else:
            spacing = 0
        for i in range(self.schedule['Users']):
            if i > 0 and spacing > 0:
                time.sleep(spacing)
            user = VirtualUser(self.q, self.start_time, i + 1, self.name, self.test, self.pacing, self.configuration)
            user.setDaemon(True)
            thread_refs.append(user)
            print 'starting script %s, user %i' % (self.name, i + 1)
            user.start()            
        for user in thread_refs:
            user.join()


class VirtualUser(threading.Thread):
    def __init__(self, queue, start_time, user_id, name, test, pacing, configuration={}):
        threading.Thread.__init__(self)
        self.q = queue
        self.start_time = start_time
        self.user_id = user_id
        self.name = name
        self.test = test
        self.pacing = pacing
        self.configuration = deepcopy(configuration)
        self.configuration['log'] = self.log
        self.configuration['thinktime'] = self.thinktime
        self.steps = {}
        self.iteration = 0

        # choose timer to use
        if sys.platform.startswith('win'):
            self.default_timer = time.clock
        else:
            self.default_timer = time.time
            
    def run(self):
        # here we do the Pacing
        # "Pacing":   {"Runfor": 1800, "Pause": 0},
        # "Pacing":   {"Iterations": 200, "Min": 20, "Max": 120},
        self.log('user', 'start')
        while True:
            if 'Iterations' in self.pacing:
                # Iterations
                if self.pacing['Iterations'] <= self.iteration:
                    self.log('user', 'end')
                    time.sleep(0.1)
                    break
            else:
                # Runfor
                timestamp = self.default_timer() - self.start_time 
                if timestamp >= self.pacing['Runfor']:
                    self.log('user', 'end')
                    time.sleep(0.1)
                    break
            self.iteration += 1
            start = self.default_timer() 
            try:
                tc = _instanciate_with_kwargs(self.test, self.configuration)
                tc.setup()
                self.log('iteration', 'start')  # start iteration timer after setup!
                status = 'failed' if tc.run() else 'end'
            except Exception, e:
                status = 'failed'
                print traceback.format_exc()
                # close all open steps!!!
                for step in self.steps.keys():
                    if not step in ['user', 'iteration']:
                        self.log(step, status)
            finally:
                tc.teardown()

            # report unterminated steps
            unterminated = [x for x in self.steps.keys() if not x in ['user', 'iteration']]
            if len(unterminated):
                raise StepMismatchException('Script "%s", step "%s" not terminated.' % 
                    (self.name, ', '.join(unterminated)))
            self.log('iteration', status)
            # Pause
            if 'Pause' in self.pacing:
                time.sleep(self.pacing['Pause'])
            # Min Max
            else:
                elapsed = self.default_timer() - start
                delay = self.pacing['Min'] + random() * (self.pacing['Max'] - self.pacing['Min']) - elapsed
                if delay > 0:
                    time.sleep(delay)

    def log(self, step, status):
        # step:   stepname or 'iteration'
        # status: start / end / failed
        # step is logged as <script_name>#<step_name>#<user>#<iteration>
        # ISO 8601 conform timestamps: YYYYMMDDTHHmmSS.sss+0100
        now = self.default_timer()
        ts = (datetime.fromtimestamp(now).strftime("%Y%m%dT%H%M%S.%f")[:19]
            + ('+' if time.altzone <= 0 else '-') + '%04d' % (time.altzone / -36))
        if status == 'start':
            if step in self.steps:
                raise StepMismatchException('Script "%s", step "%s" already started.' % (self.name, step))
            else:
                self.steps[step] = now
                self.q.put((ts, '%04d#%05d#%s#%s' % (self.user_id, self.iteration, self.name, step), status))
        elif status == 'failed' or status == 'end':
            if step not in self.steps:
                raise StepMismatchException('Script "%s", step "%s" ended without start.' % (self.name, step))
            else:
                elapsed = now - self.steps[step]
                self.steps.pop(step)
                self.q.put((ts, '%04d#%05d#%s#%s' % (self.user_id, self.iteration, self.name, step), status, 
                    '%.3f' % elapsed))
        else:
            raise StepMismatchException('Script "%s", step "%s" has unknown state "%s".' % (self.name, step, status))

    def thinktime(self, duration):
        if 'ThinkTimeFactor' in self.pacing:
            time.sleep(self.pacing['ThinkTimeFactor'] * duration)
        else:
            time.sleep(duration)


class ResultWriter(threading.Thread):
    def __init__(self, queue, infile):
        threading.Thread.__init__(self)
        self.q = queue
        self.infile = infile
    
    def run(self):
        while True:
            try:
                logentry = self.q.get(False)
                self.infile.write(','.join(logentry)+'\n')
                self.infile.flush()
            except Queue.Empty:
                time.sleep(.1)


def run(scenario):
    f = open(scenario['Results']['Filename'], 'w')
    q = multiprocessing.Queue()
    rw = ResultWriter(q, f)
    rw.setDaemon(True)
    rw.start()
    
    start_time = time.time()    
    
    # one Loadmodel entry:
    # {
    # "Name":   "Duck-Script",
    # "Script":   "scenarios/supercars/duckduck.py",
    # "Configuration": {"foo": "bar"},
    # "Schedule": {"Delay": 0, "Users": 2, "Rampup": 1},
    # "Pacing":   {"Runfor": 1800, "Pause": 0, "ThinkTimeFactor": 1.1}
    # },
    for l in scenario['Loadmodel']:
        configuration = {}
        if 'Configuration' in scenario:
            configuration = deepcopy(scenario['Configuration'])
        if 'Configuration' in l:
            # consolidate global and local configuration            
            for key in l['Configuration'].keys():
                configuration[key] = l['Configuration'][key] 
        scheduler = Scheduler(q, start_time, l['Name'], l['Script'], l['Schedule'], l['Pacing'], configuration)
        scheduler.start()


# some helper functions
class StepMismatchException(Exception):
    pass

class UnknownParameterException(Exception):
    pass

def _instanciate_with_kwargs(test, all_params):
    """Provide the parameters used in the function"""
    # check for alien parameters!
    for x in inspect.getargspec(test.__init__).args[1:]:
        if not x in all_params:
            raise UnknownParameterException(
                'Unknown parameter "%s" used in recipe "%s".' % 
                (x, test.__name__))
    kwargs = { x: all_params[x] for x in inspect.getargspec(test.__init__).args[1:] }
    return test(**kwargs)
