# -*- coding: utf-8 -*-
# Copyright 2014 Michael Helmling
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 3 as
# published by the Free Software Foundation
#

"""This module implements popular channel simulations used for coding systems.

All channels currently provided (i.e., BSC and AWGN) use BPSK modulation; that is,
zeros are mapped to a +1 signal and ones are mapped to -1 on the channel.
"""
from __future__ import division, print_function
from collections import OrderedDict
import numpy as np
from lpdec.persistence import JSONDecodable

__all__ = ['Channel', 'BSC', 'AWGNC', 'SignalGenerator']


class Channel(JSONDecodable):
    """Abstract parent class for a channel.

    :param int seed: Optional initial random seed value for noise generation.

    .. attribute:: snr

       A numerical parameter corresponding to the signal-to-noise ratio of the channel.
       It is not necessarily the "real" SNR, but be chosen such that an increase indicates
       decreasing error probability. This is used for plots and statistics.

    Channel noise can be generated by calling the :class:`Channel` object with a BPSK-modulated
    codeword as parameter.
    """

    def __init__(self, seed=None):
        JSONDecodable.__init__(self)
        self.seed = seed
        self.random = None
        self.resetSeed()

    def resetSeed(self):
        """Reset the random seed of the channel to its initial value."""
        self.random = np.random.RandomState(self.seed)

    def __call__(self, modulatedSignal):
        """Afflict the given input vector (a 1-dimensional unumpy array) with channel noise.

        The input should be a codeword in BPSK modulation with unit energy (map 0/1->+1/-1).
        """
        raise NotImplementedError()

    def signalGenerator(self, code, **kwargs):
        """Convenience function returning a :class:`SignalGenerator` for this channel and the
        given code. Additional keyword args are passed to the :class:`SignalGenerator` constructor.
        """
        return SignalGenerator(code, self, **kwargs)


class AWGNC(Channel):
    """The additive white Gaussian noise channel.

    :param float snr: Signal-to-noise ratio (:math:`\\frac{E_b}{N_0}`) in dB.
    :param float coderate: Coding rate (needed to calculate :math:`E_c` from :math:`E_b`.
    :param int round: `(optional)` Number of decimals to which the resulting LLR value is rounded.
    """

    def __init__(self, snr, coderate, round=None, seed=None):
        Channel.__init__(self, seed)
        self.snr = float(snr)
        self.coderate = coderate
        self.variance = 8 * coderate * 10 ** (snr / 10)
        self.mean = 4 * coderate * 10 ** (snr / 10)
        self.sigma = np.sqrt(self.variance)
        self.round = round

    def __call__(self, modulatedSignal):
        word = self.random.normal(self.mean, self.sigma, modulatedSignal.shape) * modulatedSignal
        if self.round is not None:
            return np.around(word, self.round)
        return word

    def params(self):
        parms = OrderedDict([('snr', self.snr), ('coderate', self.coderate)])
        if self.round is not None:
            parms['round'] = self.round
        if self.seed is not None:
            parms['seed'] = self.seed
        return parms


class BSC(Channel):
    """The binary symmetric channel: Indepentently flips each bit with a fixed probability.

    :param  float p: the crossover probability (:math:`0 \leq p \leq 1`)
    """

    def __init__(self, p, seed=None):
        """Initialize a BSC with the given crossover probability *p*."""
        Channel.__init__(self, seed)
        self.p = p
        self.snr = 1 - p

    def __call__(self, input):
        """Apply channel
        :param np.ndarray input: output of the modulator
        """
        noise = self.random.random_sample(input.shape) < self.p
        return (input * (1 - 2 * noise)).astype(np.double)

    def params(self):
        parms = [('p', self.p)]
        if self.seed is not None:
            parms.append(('seed', self.seed))
        return OrderedDict(parms)


class SignalGenerator(object):
    """An iterator class for generating noisy channel output of transmitted codewords.

    Codewords are sent through a channel which adds noise. A random seed may be provided that is
    used for generating codewords.

    The :class:`SignalGenerator` can be used as an iterator; every call to :func:`next` will yield
    another noisy signal. Afterwards, the attributes :attr:`encoderOutput`, :attr:`channelInput`
    and :attr:`channelOutput` are available and correspond to the output signal.

    :param BinaryLinearBlockCode code: the code used
    :param Channel channel: channel model
    :param int wordSeed: (optional) random seed for generating codewords. The default value of
        ``None`` uses a random seed. The special value ``-1`` indicates that all-zero simulation
        should be performed instead.
    """
    def __init__(self, code, channel, wordSeed=None):
        self.code = code
        self.channel = channel
        self.allZero = (wordSeed == -1)
        self.channelOutput = self.channelInput = self.encoderOutput = None
        if self.allZero:
            self.channelInput = np.ones(code.blocklength, dtype=np.double)
            self.encoderOutput = np.zeros(code.blocklength, dtype=np.int)
        else:
            self.wordSeed = wordSeed
            self.wordRandom = np.random.RandomState(wordSeed)

    def __iter__(self):
        return self

    def next(self):
        """Generate and return the next noisy signal."""
        return self.__next__()

    def skip(self, num):
        """Skip the next `num` signals. When resuming a seeded computation, this is much more
        efficient than calling :func:`next` `num` times.
        """
        zero = np.zeros(self.code.blocklength, dtype=np.int)
        for _ in xrange(num):
            if not self.allZero:
                self.wordRandom.randint(0, 2, self.code.infolength)
            self.channel(zero)
    
    def __next__(self):
        if not self.allZero:
            infoWord = self.wordRandom.randint(0, 2, self.code.infolength)
            self.encoderOutput = self.code.encode(infoWord)
            self.channelInput = 1 - 2 * self.encoderOutput  # BPSK
        self.channelOutput = self.channel(self.channelInput)
        return self.channelOutput

    def correctObjectiveValue(self):
        """Returns the objective value (scalar product of :attr:`channelOutput` and
        :attr:`encoderOutput`) of the codeword that was actually sent.
        """
        return np.dot(self.encoderOutput, self.channelOutput)
