#!/usr/bin/python -W default
#
import sys
import time

import lrparsing
from lrparsing import (
        Choice, Grammar, Keyword, List, Opt, Prio, Repeat, Ref, Right, Some,
        Token, Tokens, TokenRegistry)

class HtmlGrammar(Grammar):

    class T(TokenRegistry):
	tag_name = Token(re="[^ \t\r\n<>='\"/]+", case=False)
	string = Token(re="'[^''''''']*'|" + '"[^"]*"')
	cdata = Token(re='<!CDATA\\[\\[(?:[^]]|\\][^]]|\\]\\][^>])*\]\]>')
	comment = Token(re='<!--(?:[^-]|-[^-]|--[^>])*-->')
        text = Token(re='(?<=[>])[^<>]+(?=[<])')
        whitespace = Token(re="[ \t\r\n]+")
        text_start = Token(re="\\A[^<>]+(?=[<])")
        text_end = Token(re="(?<=[>])[^<>]+\\Z")

        def tokeniser(self, *args):
            in_elt = False
            is_tag = False
            for it in super(HtmlGrammar.T, self).tokeniser(*args):
                token = it[0]
                data = it[1]
                if not is_tag:
                    if token.owner == self.tag_name and token != self.tag_name:
                        it = (self.tag_name,) + it[1:]
                if data[0] == '<':
                    is_tag = True
                    in_elt = True
                else:
                    is_tag = False
                    if data[-1] == '>':
                        in_elt = False
                    elif not in_elt and token is not self.cdata and token is not self.text_start and token is not self.text_end:
                        it = (self.text,) + it[1:]
                yield it

    element = Ref("element")
    opt_space = Opt(T.whitespace)
    attr = (
            T.tag_name + 
            Opt(opt_space + Token('=') + opt_space + (T.tag_name | T.string)))
    attrs = T.whitespace + List(attr, T.whitespace, 1) + opt_space | opt_space
    nested = Repeat(element | T.text | T.whitespace)

    def make_element(name, opt_space=opt_space, nested=nested, attrs=attrs):
        return (
                Token('<') + opt_space + name + attrs + Tokens('> >/') <<
                nested <<
                Token('</') + opt_space + name + attrs + Tokens('>', '/>'))

    def make_prio(prio_tags, make_element=make_element):
        tags = (Keyword(tag, case=False) for tag in prio_tags.split())
        return Choice(*[make_element(keyword) for keyword in tags])

    directive = (
            Token('<!') + T.tag_name +
            Repeat(T.whitespace + (T.tag_name | T.string)) +
            Opt(T.whitespace) + Tokens("> />"))
    
    element = Prio(
            directive,
            Token('<') + opt_space + T.tag_name + attrs + Tokens('> />'),
            Token("</") + opt_space + T.tag_name + attrs + Tokens('>', '/>'),
            make_prio("object blockquote"),
            make_prio("i big b tt pre address em strong dfn code samp kbd var cite abbr acronym"),
            make_prio("a"),
            make_prio("h1 h2 h3 h4 h5 h6 p title script"),
            make_prio("li dt dd"),
            make_prio("ul ol dl"),
            make_prio("div span"),
            make_prio("td"),
            make_prio("colgroup tr"),
            make_prio("tbody th thead"),
            make_prio("form table"),
            make_prio("iframe"),
            make_prio("frame"),
            make_prio("head body"),
            make_prio("html"))

    START = Opt(T.text_start) + nested + Opt(T.text_end)

    WHITESPACE = ""
    COMMENTS = T.comment

def main(argv=sys.argv):
    start_time = time.time()
    if len(argv) == 2 and argv[1] == '--compile':
        pre_compiled = None
    else:
        pre_compiled = HtmlGrammar.pre_compiled
    try:
        pre_comp = HtmlGrammar.pre_compile_grammar(pre_compiled)
    except lrparsing.GrammarError, e:
        print HtmlGrammar.repr_grammar()
        print HtmlGrammar.repr_productions()
        print HtmlGrammar.repr_parse_table()
        raise
    if pre_comp is not None:
        print "compile time: %f secs" % (time.time() - start_time)
        print "Please edit set %r and set %s.pre_compiled to:" % (__file__, HtmlGrammar.__name__)
        print pre_comp
        sys.exit(1)
    quiet = False
    if len(argv) >= 2 and argv[1] == '--quiet':
        quiet = True
        argv[1:] = argv[2:]
    if len(argv) < 2:
        sys.stderr.write("usage: %s file.html ...\n" % argv[0])
        sys.exit(1)
    log = lambda l: sys.stdout.write(l+"\n")
    if len(argv) == 2:
        filename = argv[1]
        parse_tree = HtmlGrammar.parse(open(filename).read())
        print HtmlGrammar.repr_parse_tree(parse_tree)
    else:
        for filename in argv[1:]:
            print filename
            print "=" * len(filename)
            print
            parse_tree = HtmlGrammar.parse(open(filename).read())
            print HtmlGrammar.repr_parse_tree(parse_tree)

HtmlGrammar.pre_compiled = ('1a144e5b12ce4af057ee673b4bee0d76d23b8c5afcc46f22eb13f6eb4cdfbc56d79f108c9e7ece353c17ebae099912612d1571f074ef61a63d02f5f8f9059612', 0, ({'T.text': 6, '__end_of_input__': (4, 0, 'nested'), 'T.whitespace': 7, "'<!'": 79, 'T.text_end': (4, 0, 'nested'), 'T.text_start': 3, "'<'": 28, "'</'": 29}, {2: 1, 4: 2, 7: 4, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['<HtmlGrammar>']), ({'__end_of_input__': 80}, {}, ['<HtmlGrammar>']), ({'__end_of_input__': (2, 1, 'START'), 'T.text_end': 81}, {}, ['START']), ({'T.text': 6, '__end_of_input__': (4, 0, 'nested'), 'T.whitespace': 7, "'<!'": 79, 'T.text_end': (4, 0, 'nested'), "'</'": 29, "'<'": 28}, {4: 82, 7: 4, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['START']), ({'T.text': 84, '__end_of_input__': (4, 1, 'nested'), 'T.whitespace': 85, "'<!'": 79, 'T.text_end': (4, 1, 'nested'), "'</'": 29, "'<'": 28}, {8: 83, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['nested']), ({'T.text': (7, 1), '__end_of_input__': (7, 1), 'T.whitespace': (7, 1), "'<!'": (7, 1), 'T.text_end': (7, 1), "'</'": (7, 1), "'<'": (7, 1)}, {}, ['nested']), ({'T.text': (7, 1), '__end_of_input__': (7, 1), 'T.whitespace': (7, 1), "'<!'": (7, 1), 'T.text_end': (7, 1), "'</'": (7, 1), "'<'": (7, 1)}, {}, ['nested']), ({'T.text': (7, 1), '__end_of_input__': (7, 1), 'T.whitespace': (7, 1), "'<!'": (7, 1), 'T.text_end': (7, 1), "'</'": (7, 1), "'<'": (7, 1)}, {}, ['nested']), ({'T.text': (8, 1, 'element'), '__end_of_input__': (8, 1, 'element'), 'T.whitespace': (8, 1, 'element'), "'<!'": (8, 1, 'element'), 'T.text_end': (8, 1, 'element'), "'</'": (8, 1, 'element'), "'<'": (8, 1, 'element')}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (11, 1), '__end_of_input__': (11, 1), 'T.whitespace': (11, 1), "'<!'": (11, 1), 'T.text_end': (11, 1), "'</'": (11, 1), "'<'": (11, 1)}, {}, ['element']), ({'T.text': (12, 1), '__end_of_input__': (12, 1), 'T.whitespace': (12, 1), "'<!'": (12, 1), 'T.text_end': (12, 1), "'</'": (12, 1), "'<'": (12, 1)}, {}, ['element']), ({"'form'": (32, 0, 'opt_space'), "'code'": (32, 0, 'opt_space'), "'kbd'": (32, 0, 'opt_space'), "'html'": (32, 0, 'opt_space'), 'T.whitespace': 86, 'T.tag_name': (32, 0, 'opt_space'), "'h3'": (32, 0, 'opt_space'), "'dfn'": (32, 0, 'opt_space'), "'object'": (32, 0, 'opt_space'), "'acronym'": (32, 0, 'opt_space'), "'iframe'": (32, 0, 'opt_space'), "'h5'": (32, 0, 'opt_space'), "'tr'": (32, 0, 'opt_space'), "'th'": (32, 0, 'opt_space'), "'colgroup'": (32, 0, 'opt_space'), "'ul'": (32, 0, 'opt_space'), "'thead'": (32, 0, 'opt_space'), "'p'": (32, 0, 'opt_space'), "'abbr'": (32, 0, 'opt_space'), "'i'": (32, 0, 'opt_space'), "'em'": (32, 0, 'opt_space'), "'dt'": (32, 0, 'opt_space'), "'ol'": (32, 0, 'opt_space'), "'title'": (32, 0, 'opt_space'), "'strong'": (32, 0, 'opt_space'), "'div'": (32, 0, 'opt_space'), "'h6'": (32, 0, 'opt_space'), "'tt'": (32, 0, 'opt_space'), "'tbody'": (32, 0, 'opt_space'), "'body'": (32, 0, 'opt_space'), "'li'": (32, 0, 'opt_space'), "'frame'": (32, 0, 'opt_space'), "'h4'": (32, 0, 'opt_space'), "'big'": (32, 0, 'opt_space'), "'h2'": (32, 0, 'opt_space'), "'head'": (32, 0, 'opt_space'), "'cite'": (32, 0, 'opt_space'), "'span'": (32, 0, 'opt_space'), "'pre'": (32, 0, 'opt_space'), "'script'": (32, 0, 'opt_space'), "'td'": (32, 0, 'opt_space'), "'a'": (32, 0, 'opt_space'), "'blockquote'": (32, 0, 'opt_space'), "'b'": (32, 0, 'opt_space'), "'samp'": (32, 0, 'opt_space'), "'dd'": (32, 0, 'opt_space'), "'h1'": (32, 0, 'opt_space'), "'address'": (32, 0, 'opt_space'), "'table'": (32, 0, 'opt_space'), "'var'": (32, 0, 'opt_space'), "'dl'": (32, 0, 'opt_space')}, {32: 87}, ['element']), ({'T.whitespace': 86, 'T.tag_name': (32, 0, 'opt_space')}, {32: 88}, ['element']), ({'T.text': (15, 1), '__end_of_input__': (15, 1), 'T.whitespace': (15, 1), "'<!'": (15, 1), 'T.text_end': (15, 1), "'</'": (15, 1), "'<'": (15, 1)}, {}, ['element']), ({'T.text': (15, 1), '__end_of_input__': (15, 1), 'T.whitespace': (15, 1), "'<!'": (15, 1), 'T.text_end': (15, 1), "'</'": (15, 1), "'<'": (15, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (16, 1), '__end_of_input__': (16, 1), 'T.whitespace': (16, 1), "'<!'": (16, 1), 'T.text_end': (16, 1), "'</'": (16, 1), "'<'": (16, 1)}, {}, ['element']), ({'T.text': (17, 1), '__end_of_input__': (17, 1), 'T.whitespace': (17, 1), "'<!'": (17, 1), 'T.text_end': (17, 1), "'</'": (17, 1), "'<'": (17, 1)}, {}, ['element']), ({'T.text': (18, 1), '__end_of_input__': (18, 1), 'T.whitespace': (18, 1), "'<!'": (18, 1), 'T.text_end': (18, 1), "'</'": (18, 1), "'<'": (18, 1)}, {}, ['element']), ({'T.text': (18, 1), '__end_of_input__': (18, 1), 'T.whitespace': (18, 1), "'<!'": (18, 1), 'T.text_end': (18, 1), "'</'": (18, 1), "'<'": (18, 1)}, {}, ['element']), ({'T.text': (18, 1), '__end_of_input__': (18, 1), 'T.whitespace': (18, 1), "'<!'": (18, 1), 'T.text_end': (18, 1), "'</'": (18, 1), "'<'": (18, 1)}, {}, ['element']), ({'T.text': (18, 1), '__end_of_input__': (18, 1), 'T.whitespace': (18, 1), "'<!'": (18, 1), 'T.text_end': (18, 1), "'</'": (18, 1), "'<'": (18, 1)}, {}, ['element']), ({'T.text': (18, 1), '__end_of_input__': (18, 1), 'T.whitespace': (18, 1), "'<!'": (18, 1), 'T.text_end': (18, 1), "'</'": (18, 1), "'<'": (18, 1)}, {}, ['element']), ({'T.text': (18, 1), '__end_of_input__': (18, 1), 'T.whitespace': (18, 1), "'<!'": (18, 1), 'T.text_end': (18, 1), "'</'": (18, 1), "'<'": (18, 1)}, {}, ['element']), ({'T.text': (18, 1), '__end_of_input__': (18, 1), 'T.whitespace': (18, 1), "'<!'": (18, 1), 'T.text_end': (18, 1), "'</'": (18, 1), "'<'": (18, 1)}, {}, ['element']), ({'T.text': (18, 1), '__end_of_input__': (18, 1), 'T.whitespace': (18, 1), "'<!'": (18, 1), 'T.text_end': (18, 1), "'</'": (18, 1), "'<'": (18, 1)}, {}, ['element']), ({'T.text': (18, 1), '__end_of_input__': (18, 1), 'T.whitespace': (18, 1), "'<!'": (18, 1), 'T.text_end': (18, 1), "'</'": (18, 1), "'<'": (18, 1)}, {}, ['element']), ({'T.text': (19, 1), '__end_of_input__': (19, 1), 'T.whitespace': (19, 1), "'<!'": (19, 1), 'T.text_end': (19, 1), "'</'": (19, 1), "'<'": (19, 1)}, {}, ['element']), ({'T.text': (19, 1), '__end_of_input__': (19, 1), 'T.whitespace': (19, 1), "'<!'": (19, 1), 'T.text_end': (19, 1), "'</'": (19, 1), "'<'": (19, 1)}, {}, ['element']), ({'T.text': (19, 1), '__end_of_input__': (19, 1), 'T.whitespace': (19, 1), "'<!'": (19, 1), 'T.text_end': (19, 1), "'</'": (19, 1), "'<'": (19, 1)}, {}, ['element']), ({'T.text': (20, 1), '__end_of_input__': (20, 1), 'T.whitespace': (20, 1), "'<!'": (20, 1), 'T.text_end': (20, 1), "'</'": (20, 1), "'<'": (20, 1)}, {}, ['element']), ({'T.text': (20, 1), '__end_of_input__': (20, 1), 'T.whitespace': (20, 1), "'<!'": (20, 1), 'T.text_end': (20, 1), "'</'": (20, 1), "'<'": (20, 1)}, {}, ['element']), ({'T.text': (20, 1), '__end_of_input__': (20, 1), 'T.whitespace': (20, 1), "'<!'": (20, 1), 'T.text_end': (20, 1), "'</'": (20, 1), "'<'": (20, 1)}, {}, ['element']), ({'T.text': (21, 1), '__end_of_input__': (21, 1), 'T.whitespace': (21, 1), "'<!'": (21, 1), 'T.text_end': (21, 1), "'</'": (21, 1), "'<'": (21, 1)}, {}, ['element']), ({'T.text': (21, 1), '__end_of_input__': (21, 1), 'T.whitespace': (21, 1), "'<!'": (21, 1), 'T.text_end': (21, 1), "'</'": (21, 1), "'<'": (21, 1)}, {}, ['element']), ({'T.text': (22, 1), '__end_of_input__': (22, 1), 'T.whitespace': (22, 1), "'<!'": (22, 1), 'T.text_end': (22, 1), "'</'": (22, 1), "'<'": (22, 1)}, {}, ['element']), ({'T.text': (23, 1), '__end_of_input__': (23, 1), 'T.whitespace': (23, 1), "'<!'": (23, 1), 'T.text_end': (23, 1), "'</'": (23, 1), "'<'": (23, 1)}, {}, ['element']), ({'T.text': (23, 1), '__end_of_input__': (23, 1), 'T.whitespace': (23, 1), "'<!'": (23, 1), 'T.text_end': (23, 1), "'</'": (23, 1), "'<'": (23, 1)}, {}, ['element']), ({'T.text': (24, 1), '__end_of_input__': (24, 1), 'T.whitespace': (24, 1), "'<!'": (24, 1), 'T.text_end': (24, 1), "'</'": (24, 1), "'<'": (24, 1)}, {}, ['element']), ({'T.text': (24, 1), '__end_of_input__': (24, 1), 'T.whitespace': (24, 1), "'<!'": (24, 1), 'T.text_end': (24, 1), "'</'": (24, 1), "'<'": (24, 1)}, {}, ['element']), ({'T.text': (24, 1), '__end_of_input__': (24, 1), 'T.whitespace': (24, 1), "'<!'": (24, 1), 'T.text_end': (24, 1), "'</'": (24, 1), "'<'": (24, 1)}, {}, ['element']), ({'T.text': (25, 1), '__end_of_input__': (25, 1), 'T.whitespace': (25, 1), "'<!'": (25, 1), 'T.text_end': (25, 1), "'</'": (25, 1), "'<'": (25, 1)}, {}, ['element']), ({'T.text': (25, 1), '__end_of_input__': (25, 1), 'T.whitespace': (25, 1), "'<!'": (25, 1), 'T.text_end': (25, 1), "'</'": (25, 1), "'<'": (25, 1)}, {}, ['element']), ({'T.text': (26, 1), '__end_of_input__': (26, 1), 'T.whitespace': (26, 1), "'<!'": (26, 1), 'T.text_end': (26, 1), "'</'": (26, 1), "'<'": (26, 1)}, {}, ['element']), ({'T.text': (27, 1), '__end_of_input__': (27, 1), 'T.whitespace': (27, 1), "'<!'": (27, 1), 'T.text_end': (27, 1), "'</'": (27, 1), "'<'": (27, 1)}, {}, ['element']), ({'T.text': (28, 1), '__end_of_input__': (28, 1), 'T.whitespace': (28, 1), "'<!'": (28, 1), 'T.text_end': (28, 1), "'</'": (28, 1), "'<'": (28, 1)}, {}, ['element']), ({'T.text': (28, 1), '__end_of_input__': (28, 1), 'T.whitespace': (28, 1), "'<!'": (28, 1), 'T.text_end': (28, 1), "'</'": (28, 1), "'<'": (28, 1)}, {}, ['element']), ({'T.text': (29, 1), '__end_of_input__': (29, 1), 'T.whitespace': (29, 1), "'<!'": (29, 1), 'T.text_end': (29, 1), "'</'": (29, 1), "'<'": (29, 1)}, {}, ['element']), ({'T.tag_name': 89}, {}, ['directive']), ({'__empty__': (0, 2, '<HtmlGrammar>')}, {}, ['<HtmlGrammar>']), ({'__end_of_input__': (2, 2, 'START')}, {}, ['START']), ({'__end_of_input__': (2, 2, 'START'), 'T.text_end': 90}, {}, ['START']), ({'T.text': (7, 2), '__end_of_input__': (7, 2), 'T.whitespace': (7, 2), "'<!'": (7, 2), 'T.text_end': (7, 2), "'</'": (7, 2), "'<'": (7, 2)}, {}, ['nested']), ({'T.text': (7, 2), '__end_of_input__': (7, 2), 'T.whitespace': (7, 2), "'<!'": (7, 2), 'T.text_end': (7, 2), "'</'": (7, 2), "'<'": (7, 2)}, {}, ['nested']), ({'T.text': (7, 2), '__end_of_input__': (7, 2), 'T.whitespace': (7, 2), "'<!'": (7, 2), 'T.text_end': (7, 2), "'</'": (7, 2), "'<'": (7, 2)}, {}, ['nested']), ({"'ol'": (32, 1, 'opt_space'), "'tt'": (32, 1, 'opt_space'), "'h5'": (32, 1, 'opt_space'), "'h3'": (32, 1, 'opt_space'), "'dfn'": (32, 1, 'opt_space'), "'kbd'": (32, 1, 'opt_space'), "'object'": (32, 1, 'opt_space'), "'body'": (32, 1, 'opt_space'), "'acronym'": (32, 1, 'opt_space'), "'iframe'": (32, 1, 'opt_space'), "'code'": (32, 1, 'opt_space'), "'tr'": (32, 1, 'opt_space'), "'th'": (32, 1, 'opt_space'), 'T.tag_name': (32, 1, 'opt_space'), "'colgroup'": (32, 1, 'opt_space'), "'ul'": (32, 1, 'opt_space'), "'thead'": (32, 1, 'opt_space'), "'p'": (32, 1, 'opt_space'), "'abbr'": (32, 1, 'opt_space'), "'i'": (32, 1, 'opt_space'), "'em'": (32, 1, 'opt_space'), "'dt'": (32, 1, 'opt_space'), "'title'": (32, 1, 'opt_space'), "'strong'": (32, 1, 'opt_space'), "'div'": (32, 1, 'opt_space'), "'h6'": (32, 1, 'opt_space'), "'html'": (32, 1, 'opt_space'), "'tbody'": (32, 1, 'opt_space'), "'td'": (32, 1, 'opt_space'), "'li'": (32, 1, 'opt_space'), "'frame'": (32, 1, 'opt_space'), "'h4'": (32, 1, 'opt_space'), "'big'": (32, 1, 'opt_space'), "'h2'": (32, 1, 'opt_space'), "'head'": (32, 1, 'opt_space'), "'cite'": (32, 1, 'opt_space'), "'span'": (32, 1, 'opt_space'), "'pre'": (32, 1, 'opt_space'), "'script'": (32, 1, 'opt_space'), "'form'": (32, 1, 'opt_space'), "'a'": (32, 1, 'opt_space'), "'blockquote'": (32, 1, 'opt_space'), "'b'": (32, 1, 'opt_space'), "'samp'": (32, 1, 'opt_space'), "'dd'": (32, 1, 'opt_space'), "'h1'": (32, 1, 'opt_space'), "'='": (32, 1, 'opt_space'), "'address'": (32, 1, 'opt_space'), "'table'": (32, 1, 'opt_space'), "'var'": (32, 1, 'opt_space'), "'dl'": (32, 1, 'opt_space')}, {}, ['opt_space']), ({"'code'": 103, "'kbd'": 105, 'T.tag_name': 91, "'h3'": 113, "'dfn'": 102, "'object'": 92, "'body'": 139, "'acronym'": 109, "'h5'": 115, "'iframe'": 136, "'ol'": 124, "'tr'": 130, "'head'": 138, "'tt'": 97, "'ul'": 123, "'thead'": 133, "'p'": 117, "'abbr'": 108, "'i'": 94, "'em'": 100, "'dt'": 121, "'title'": 118, "'strong'": 101, "'blockquote'": 93, "'h6'": 116, "'html'": 140, "'tbody'": 131, "'td'": 128, "'li'": 120, "'frame'": 137, "'div'": 126, "'big'": 95, "'h2'": 112, "'th'": 132, "'cite'": 107, "'span'": 127, "'pre'": 98, "'script'": 119, "'form'": 134, "'a'": 110, "'colgroup'": 129, "'b'": 96, "'samp'": 104, "'dd'": 122, "'h1'": 111, "'var'": 106, "'address'": 99, "'table'": 135, "'h4'": 114, "'dl'": 125}, {}, ['element']), ({'T.tag_name': 141}, {}, ['element']), ({'T.whitespace': 142, "'/>'": 144, "'>'": 143}, {88: 145}, ['directive']), ({'__end_of_input__': (2, 3, 'START')}, {}, ['START']), ({'T.whitespace': 146, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 147, 34: 148}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 149}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 153}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 154}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 157}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 158}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 159}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 160}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 161}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 162}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 163}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 164}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 165}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 166}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 167}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 168}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 169}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 170}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 171}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 172}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 173}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 174}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 175}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 176}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 177}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 178}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 179}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 180}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 181}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 182}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 183}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 184}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 185}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 186}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 187}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 188}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 189}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 190}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 191}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 192}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 193}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 194}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 195}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 196}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 197}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 198}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 199}, ['element']), ({'T.whitespace': 146, "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 147, 34: 200}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 201}, ['element']), ({"'>/'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), 'T.whitespace': 146}, {32: 147, 34: 202}, ['element']), ({'T.whitespace': 146, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 147, 34: 203}, ['element']), ({'T.tag_name': 204, "'/>'": 206, "'>'": 205, 'T.string': 207}, {}, ['directive']), ({'T.text': (30, 3, 'directive'), '__end_of_input__': (30, 3, 'directive'), 'T.whitespace': (30, 3, 'directive'), "'<!'": (30, 3, 'directive'), 'T.text_end': (30, 3, 'directive'), "'</'": (30, 3, 'directive'), "'<'": (30, 3, 'directive')}, {}, ['directive']), ({'T.text': (30, 3, 'directive'), '__end_of_input__': (30, 3, 'directive'), 'T.whitespace': (30, 3, 'directive'), "'<!'": (30, 3, 'directive'), 'T.text_end': (30, 3, 'directive'), "'</'": (30, 3, 'directive'), "'<'": (30, 3, 'directive')}, {}, ['directive']), ({'T.whitespace': 208, "'/>'": 210, "'>'": 209}, {}, ['directive']), ({"'>/'": (32, 1, 'opt_space'), 'T.tag_name': 150, "'/>'": (32, 1, 'opt_space'), "'>'": (32, 1, 'opt_space')}, {89: 151, 141: 152}, ['attrs', 'opt_space']), ({"'>/'": (34, 1, 'attrs'), "'/>'": (34, 1, 'attrs'), "'>'": (34, 1, 'attrs')}, {}, ['attrs']), ({"'/>'": 212, "'>'": 211}, {}, ['element']), ({"'>/'": 214, "'>'": 213}, {}, ['element']), ({'T.whitespace': (141, 1, 'attr'), "'/>'": (141, 1, 'attr'), "'='": (32, 0, 'opt_space'), "'>'": (141, 1, 'attr'), "'>/'": (141, 1, 'attr')}, {32: 215}, ['attr']), ({'T.whitespace': 155, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space'), "'>/'": (32, 0, 'opt_space')}, {32: 156}, ['attrs']), ({'T.whitespace': (89, 1), "'/>'": (89, 1), "'>'": (89, 1), "'>/'": (89, 1)}, {}, ['attrs']), ({"'>/'": 217, "'>'": 216}, {}, ['element']), ({"'>/'": 219, "'>'": 218}, {}, ['element']), ({"'>/'": (32, 1, 'opt_space'), 'T.tag_name': 220, "'/>'": (32, 1, 'opt_space'), "'>'": (32, 1, 'opt_space')}, {141: 221}, ['attrs', 'opt_space']), ({"'>/'": (34, 3, 'attrs'), "'/>'": (34, 3, 'attrs'), "'>'": (34, 3, 'attrs')}, {}, ['attrs']), ({"'>/'": 223, "'>'": 222}, {}, ['element']), ({"'>/'": 225, "'>'": 224}, {}, ['element']), ({"'>/'": 227, "'>'": 226}, {}, ['element']), ({"'>/'": 229, "'>'": 228}, {}, ['element']), ({"'>/'": 231, "'>'": 230}, {}, ['element']), ({"'>/'": 233, "'>'": 232}, {}, ['element']), ({"'>/'": 235, "'>'": 234}, {}, ['element']), ({"'>/'": 237, "'>'": 236}, {}, ['element']), ({"'>/'": 239, "'>'": 238}, {}, ['element']), ({"'>/'": 241, "'>'": 240}, {}, ['element']), ({"'>/'": 243, "'>'": 242}, {}, ['element']), ({"'>/'": 245, "'>'": 244}, {}, ['element']), ({"'>/'": 247, "'>'": 246}, {}, ['element']), ({"'>/'": 249, "'>'": 248}, {}, ['element']), ({"'>/'": 251, "'>'": 250}, {}, ['element']), ({"'>/'": 253, "'>'": 252}, {}, ['element']), ({"'>/'": 255, "'>'": 254}, {}, ['element']), ({"'>/'": 257, "'>'": 256}, {}, ['element']), ({"'>/'": 259, "'>'": 258}, {}, ['element']), ({"'>/'": 261, "'>'": 260}, {}, ['element']), ({"'>/'": 263, "'>'": 262}, {}, ['element']), ({"'>/'": 265, "'>'": 264}, {}, ['element']), ({"'>/'": 267, "'>'": 266}, {}, ['element']), ({"'>/'": 269, "'>'": 268}, {}, ['element']), ({"'>/'": 271, "'>'": 270}, {}, ['element']), ({"'>/'": 273, "'>'": 272}, {}, ['element']), ({"'>/'": 275, "'>'": 274}, {}, ['element']), ({"'>/'": 277, "'>'": 276}, {}, ['element']), ({"'>/'": 279, "'>'": 278}, {}, ['element']), ({"'>/'": 281, "'>'": 280}, {}, ['element']), ({"'>/'": 283, "'>'": 282}, {}, ['element']), ({"'>/'": 285, "'>'": 284}, {}, ['element']), ({"'>/'": 287, "'>'": 286}, {}, ['element']), ({"'>/'": 289, "'>'": 288}, {}, ['element']), ({"'>/'": 291, "'>'": 290}, {}, ['element']), ({"'>/'": 293, "'>'": 292}, {}, ['element']), ({"'>/'": 295, "'>'": 294}, {}, ['element']), ({"'>/'": 297, "'>'": 296}, {}, ['element']), ({"'>/'": 299, "'>'": 298}, {}, ['element']), ({"'>/'": 301, "'>'": 300}, {}, ['element']), ({"'>/'": 303, "'>'": 302}, {}, ['element']), ({"'>/'": 305, "'>'": 304}, {}, ['element']), ({"'>/'": 307, "'>'": 306}, {}, ['element']), ({"'>/'": 309, "'>'": 308}, {}, ['element']), ({"'>/'": 311, "'>'": 310}, {}, ['element']), ({"'>/'": 313, "'>'": 312}, {}, ['element']), ({"'/>'": 315, "'>'": 314}, {}, ['element']), ({'T.whitespace': (88, 2), "'/>'": (88, 2), "'>'": (88, 2)}, {}, ['directive']), ({'T.text': (30, 4, 'directive'), '__end_of_input__': (30, 4, 'directive'), 'T.whitespace': (30, 4, 'directive'), "'<!'": (30, 4, 'directive'), 'T.text_end': (30, 4, 'directive'), "'</'": (30, 4, 'directive'), "'<'": (30, 4, 'directive')}, {}, ['directive']), ({'T.text': (30, 4, 'directive'), '__end_of_input__': (30, 4, 'directive'), 'T.whitespace': (30, 4, 'directive'), "'<!'": (30, 4, 'directive'), 'T.text_end': (30, 4, 'directive'), "'</'": (30, 4, 'directive'), "'<'": (30, 4, 'directive')}, {}, ['directive']), ({'T.whitespace': (88, 2), "'/>'": (88, 2), "'>'": (88, 2)}, {}, ['directive']), ({'T.tag_name': 316, "'/>'": 318, "'>'": 317, 'T.string': 319}, {}, ['directive']), ({'T.text': (30, 4, 'directive'), '__end_of_input__': (30, 4, 'directive'), 'T.whitespace': (30, 4, 'directive'), "'<!'": (30, 4, 'directive'), 'T.text_end': (30, 4, 'directive'), "'</'": (30, 4, 'directive'), "'<'": (30, 4, 'directive')}, {}, ['directive']), ({'T.text': (30, 4, 'directive'), '__end_of_input__': (30, 4, 'directive'), 'T.whitespace': (30, 4, 'directive'), "'<!'": (30, 4, 'directive'), 'T.text_end': (30, 4, 'directive'), "'</'": (30, 4, 'directive'), "'<'": (30, 4, 'directive')}, {}, ['directive']), ({'T.text': (13, 5), '__end_of_input__': (13, 5), 'T.whitespace': (13, 5), "'<!'": (13, 5), 'T.text_end': (13, 5), "'</'": (13, 5), "'<'": (13, 5)}, {}, ['element']), ({'T.text': (13, 5), '__end_of_input__': (13, 5), 'T.whitespace': (13, 5), "'<!'": (13, 5), 'T.text_end': (13, 5), "'</'": (13, 5), "'<'": (13, 5)}, {}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 320, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 322, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({"'='": 323}, {}, ['attr']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 324, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 325, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 326, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 327, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': (141, 1, 'attr'), "'/>'": (141, 1, 'attr'), "'='": (32, 0, 'opt_space'), "'>'": (141, 1, 'attr'), "'>/'": (141, 1, 'attr')}, {32: 328}, ['attr']), ({'T.whitespace': (89, 3), "'/>'": (89, 3), "'>'": (89, 3), "'>/'": (89, 3)}, {}, ['attrs']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 329, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 330, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 331, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 332, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 333, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 334, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 335, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 336, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 337, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 338, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 339, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 340, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 341, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 342, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 343, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 344, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 345, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 346, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 347, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 348, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 349, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 350, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 351, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 352, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 353, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 354, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 355, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 356, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 357, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 358, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 359, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 360, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 361, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 362, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 363, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 364, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 365, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 366, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 367, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 368, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 369, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 370, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 371, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 372, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 373, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 374, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 375, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 376, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 377, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 378, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 379, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 380, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 381, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 382, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 383, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 384, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 385, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 386, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 387, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 388, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 389, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 390, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 391, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 392, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 393, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 394, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 395, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 396, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 397, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 398, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 399, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 400, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 401, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 402, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 403, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 404, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 405, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 406, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 407, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 408, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 409, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 410, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 411, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 412, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 413, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 414, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 415, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 416, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 417, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 418, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 419, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.whitespace': 7, 'T.text': 6, "'</'": (4, 0, 'nested'), "'<!'": 79, "'<'": 28}, {4: 420, 7: 321, 8: 5, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['element']), ({'T.text': (14, 5), '__end_of_input__': (14, 5), 'T.whitespace': (14, 5), "'<!'": (14, 5), 'T.text_end': (14, 5), "'</'": (14, 5), "'<'": (14, 5)}, {}, ['element']), ({'T.text': (14, 5), '__end_of_input__': (14, 5), 'T.whitespace': (14, 5), "'<!'": (14, 5), 'T.text_end': (14, 5), "'</'": (14, 5), "'<'": (14, 5)}, {}, ['element']), ({'T.whitespace': (88, 3), "'/>'": (88, 3), "'>'": (88, 3)}, {}, ['directive']), ({'T.text': (30, 5, 'directive'), '__end_of_input__': (30, 5, 'directive'), 'T.whitespace': (30, 5, 'directive'), "'<!'": (30, 5, 'directive'), 'T.text_end': (30, 5, 'directive'), "'</'": (30, 5, 'directive'), "'<'": (30, 5, 'directive')}, {}, ['directive']), ({'T.text': (30, 5, 'directive'), '__end_of_input__': (30, 5, 'directive'), 'T.whitespace': (30, 5, 'directive'), "'<!'": (30, 5, 'directive'), 'T.text_end': (30, 5, 'directive'), "'</'": (30, 5, 'directive'), "'<'": (30, 5, 'directive')}, {}, ['directive']), ({'T.whitespace': (88, 3), "'/>'": (88, 3), "'>'": (88, 3)}, {}, ['directive']), ({"'</'": 421}, {}, ['element']), ({'T.whitespace': 85, "'</'": (4, 1, 'nested'), "'<!'": 79, "'<'": 28, 'T.text': 84}, {8: 83, 11: 8, 12: 9, 13: 10, 14: 11, 15: 12, 16: 13, 17: 14, 18: 15, 19: 16, 20: 17, 21: 18, 22: 19, 23: 20, 24: 21, 25: 22, 26: 23, 27: 24, 28: 25, 29: 26, 30: 27, 38: 30, 39: 31, 40: 32, 41: 33, 42: 34, 43: 35, 44: 36, 45: 37, 46: 38, 47: 39, 48: 40, 49: 41, 50: 42, 51: 43, 52: 44, 53: 45, 54: 46, 55: 47, 56: 48, 57: 49, 58: 50, 59: 51, 60: 52, 61: 53, 62: 54, 63: 55, 64: 56, 65: 57, 66: 58, 67: 59, 68: 60, 69: 61, 70: 62, 71: 63, 72: 64, 73: 65, 74: 66, 75: 67, 76: 68, 77: 69, 78: 70, 79: 71, 80: 72, 81: 73, 82: 74, 83: 75, 84: 76, 85: 77, 86: 78}, ['nested']), ({"'</'": 422}, {}, ['element']), ({'T.whitespace': 423, 'T.tag_name': (32, 0, 'opt_space'), 'T.string': (32, 0, 'opt_space')}, {32: 424}, ['attr']), ({"'</'": 425}, {}, ['element']), ({"'</'": 426}, {}, ['element']), ({"'</'": 427}, {}, ['element']), ({"'</'": 428}, {}, ['element']), ({"'='": 429}, {}, ['attr']), ({"'</'": 430}, {}, ['element']), ({"'</'": 431}, {}, ['element']), ({"'</'": 432}, {}, ['element']), ({"'</'": 433}, {}, ['element']), ({"'</'": 434}, {}, ['element']), ({"'</'": 435}, {}, ['element']), ({"'</'": 436}, {}, ['element']), ({"'</'": 437}, {}, ['element']), ({"'</'": 438}, {}, ['element']), ({"'</'": 439}, {}, ['element']), ({"'</'": 440}, {}, ['element']), ({"'</'": 441}, {}, ['element']), ({"'</'": 442}, {}, ['element']), ({"'</'": 443}, {}, ['element']), ({"'</'": 444}, {}, ['element']), ({"'</'": 445}, {}, ['element']), ({"'</'": 446}, {}, ['element']), ({"'</'": 447}, {}, ['element']), ({"'</'": 448}, {}, ['element']), ({"'</'": 449}, {}, ['element']), ({"'</'": 450}, {}, ['element']), ({"'</'": 451}, {}, ['element']), ({"'</'": 452}, {}, ['element']), ({"'</'": 453}, {}, ['element']), ({"'</'": 454}, {}, ['element']), ({"'</'": 455}, {}, ['element']), ({"'</'": 456}, {}, ['element']), ({"'</'": 457}, {}, ['element']), ({"'</'": 458}, {}, ['element']), ({"'</'": 459}, {}, ['element']), ({"'</'": 460}, {}, ['element']), ({"'</'": 461}, {}, ['element']), ({"'</'": 462}, {}, ['element']), ({"'</'": 463}, {}, ['element']), ({"'</'": 464}, {}, ['element']), ({"'</'": 465}, {}, ['element']), ({"'</'": 466}, {}, ['element']), ({"'</'": 467}, {}, ['element']), ({"'</'": 468}, {}, ['element']), ({"'</'": 469}, {}, ['element']), ({"'</'": 470}, {}, ['element']), ({"'</'": 471}, {}, ['element']), ({"'</'": 472}, {}, ['element']), ({"'</'": 473}, {}, ['element']), ({"'</'": 474}, {}, ['element']), ({"'</'": 475}, {}, ['element']), ({"'</'": 476}, {}, ['element']), ({"'</'": 477}, {}, ['element']), ({"'</'": 478}, {}, ['element']), ({"'</'": 479}, {}, ['element']), ({"'</'": 480}, {}, ['element']), ({"'</'": 481}, {}, ['element']), ({"'</'": 482}, {}, ['element']), ({"'</'": 483}, {}, ['element']), ({"'</'": 484}, {}, ['element']), ({"'</'": 485}, {}, ['element']), ({"'</'": 486}, {}, ['element']), ({"'</'": 487}, {}, ['element']), ({"'</'": 488}, {}, ['element']), ({"'</'": 489}, {}, ['element']), ({"'</'": 490}, {}, ['element']), ({"'</'": 491}, {}, ['element']), ({"'</'": 492}, {}, ['element']), ({"'</'": 493}, {}, ['element']), ({"'</'": 494}, {}, ['element']), ({"'</'": 495}, {}, ['element']), ({"'</'": 496}, {}, ['element']), ({"'</'": 497}, {}, ['element']), ({"'</'": 498}, {}, ['element']), ({"'</'": 499}, {}, ['element']), ({"'</'": 500}, {}, ['element']), ({"'</'": 501}, {}, ['element']), ({"'</'": 502}, {}, ['element']), ({"'</'": 503}, {}, ['element']), ({"'</'": 504}, {}, ['element']), ({"'</'": 505}, {}, ['element']), ({"'</'": 506}, {}, ['element']), ({"'</'": 507}, {}, ['element']), ({"'</'": 508}, {}, ['element']), ({"'</'": 509}, {}, ['element']), ({"'</'": 510}, {}, ['element']), ({"'</'": 511}, {}, ['element']), ({"'</'": 512}, {}, ['element']), ({"'</'": 513}, {}, ['element']), ({"'</'": 514}, {}, ['element']), ({"'</'": 515}, {}, ['element']), ({"'</'": 516}, {}, ['element']), ({"'</'": 517}, {}, ['element']), ({"'</'": 518}, {}, ['element']), ({"'</'": 519}, {}, ['element']), ({"'</'": 520}, {}, ['element']), ({"'</'": 521}, {}, ['element']), ({'T.whitespace': 86, "'object'": (32, 0, 'opt_space')}, {32: 522}, ['element']), ({'T.whitespace': 86, "'object'": (32, 0, 'opt_space')}, {32: 523}, ['element']), ({'T.tag_name': (32, 1, 'opt_space'), 'T.string': (32, 1, 'opt_space')}, {}, ['opt_space']), ({'T.tag_name': 524, 'T.string': 525}, {}, ['attr']), ({'T.whitespace': 86, "'blockquote'": (32, 0, 'opt_space')}, {32: 526}, ['element']), ({'T.whitespace': 86, "'blockquote'": (32, 0, 'opt_space')}, {32: 527}, ['element']), ({"'i'": (32, 0, 'opt_space'), 'T.whitespace': 86}, {32: 528}, ['element']), ({"'i'": (32, 0, 'opt_space'), 'T.whitespace': 86}, {32: 529}, ['element']), ({'T.whitespace': 423, 'T.tag_name': (32, 0, 'opt_space'), 'T.string': (32, 0, 'opt_space')}, {32: 530}, ['attr']), ({'T.whitespace': 86, "'big'": (32, 0, 'opt_space')}, {32: 531}, ['element']), ({'T.whitespace': 86, "'big'": (32, 0, 'opt_space')}, {32: 532}, ['element']), ({'T.whitespace': 86, "'b'": (32, 0, 'opt_space')}, {32: 533}, ['element']), ({'T.whitespace': 86, "'b'": (32, 0, 'opt_space')}, {32: 534}, ['element']), ({'T.whitespace': 86, "'tt'": (32, 0, 'opt_space')}, {32: 535}, ['element']), ({'T.whitespace': 86, "'tt'": (32, 0, 'opt_space')}, {32: 536}, ['element']), ({'T.whitespace': 86, "'pre'": (32, 0, 'opt_space')}, {32: 537}, ['element']), ({'T.whitespace': 86, "'pre'": (32, 0, 'opt_space')}, {32: 538}, ['element']), ({'T.whitespace': 86, "'address'": (32, 0, 'opt_space')}, {32: 539}, ['element']), ({'T.whitespace': 86, "'address'": (32, 0, 'opt_space')}, {32: 540}, ['element']), ({'T.whitespace': 86, "'em'": (32, 0, 'opt_space')}, {32: 541}, ['element']), ({'T.whitespace': 86, "'em'": (32, 0, 'opt_space')}, {32: 542}, ['element']), ({'T.whitespace': 86, "'strong'": (32, 0, 'opt_space')}, {32: 543}, ['element']), ({'T.whitespace': 86, "'strong'": (32, 0, 'opt_space')}, {32: 544}, ['element']), ({'T.whitespace': 86, "'dfn'": (32, 0, 'opt_space')}, {32: 545}, ['element']), ({'T.whitespace': 86, "'dfn'": (32, 0, 'opt_space')}, {32: 546}, ['element']), ({'T.whitespace': 86, "'code'": (32, 0, 'opt_space')}, {32: 547}, ['element']), ({'T.whitespace': 86, "'code'": (32, 0, 'opt_space')}, {32: 548}, ['element']), ({'T.whitespace': 86, "'samp'": (32, 0, 'opt_space')}, {32: 549}, ['element']), ({'T.whitespace': 86, "'samp'": (32, 0, 'opt_space')}, {32: 550}, ['element']), ({'T.whitespace': 86, "'kbd'": (32, 0, 'opt_space')}, {32: 551}, ['element']), ({'T.whitespace': 86, "'kbd'": (32, 0, 'opt_space')}, {32: 552}, ['element']), ({'T.whitespace': 86, "'var'": (32, 0, 'opt_space')}, {32: 553}, ['element']), ({'T.whitespace': 86, "'var'": (32, 0, 'opt_space')}, {32: 554}, ['element']), ({'T.whitespace': 86, "'cite'": (32, 0, 'opt_space')}, {32: 555}, ['element']), ({'T.whitespace': 86, "'cite'": (32, 0, 'opt_space')}, {32: 556}, ['element']), ({'T.whitespace': 86, "'abbr'": (32, 0, 'opt_space')}, {32: 557}, ['element']), ({'T.whitespace': 86, "'abbr'": (32, 0, 'opt_space')}, {32: 558}, ['element']), ({'T.whitespace': 86, "'acronym'": (32, 0, 'opt_space')}, {32: 559}, ['element']), ({'T.whitespace': 86, "'acronym'": (32, 0, 'opt_space')}, {32: 560}, ['element']), ({"'a'": (32, 0, 'opt_space'), 'T.whitespace': 86}, {32: 561}, ['element']), ({"'a'": (32, 0, 'opt_space'), 'T.whitespace': 86}, {32: 562}, ['element']), ({'T.whitespace': 86, "'h1'": (32, 0, 'opt_space')}, {32: 563}, ['element']), ({'T.whitespace': 86, "'h1'": (32, 0, 'opt_space')}, {32: 564}, ['element']), ({'T.whitespace': 86, "'h2'": (32, 0, 'opt_space')}, {32: 565}, ['element']), ({'T.whitespace': 86, "'h2'": (32, 0, 'opt_space')}, {32: 566}, ['element']), ({'T.whitespace': 86, "'h3'": (32, 0, 'opt_space')}, {32: 567}, ['element']), ({'T.whitespace': 86, "'h3'": (32, 0, 'opt_space')}, {32: 568}, ['element']), ({'T.whitespace': 86, "'h4'": (32, 0, 'opt_space')}, {32: 569}, ['element']), ({'T.whitespace': 86, "'h4'": (32, 0, 'opt_space')}, {32: 570}, ['element']), ({'T.whitespace': 86, "'h5'": (32, 0, 'opt_space')}, {32: 571}, ['element']), ({'T.whitespace': 86, "'h5'": (32, 0, 'opt_space')}, {32: 572}, ['element']), ({"'h6'": (32, 0, 'opt_space'), 'T.whitespace': 86}, {32: 573}, ['element']), ({"'h6'": (32, 0, 'opt_space'), 'T.whitespace': 86}, {32: 574}, ['element']), ({'T.whitespace': 86, "'p'": (32, 0, 'opt_space')}, {32: 575}, ['element']), ({'T.whitespace': 86, "'p'": (32, 0, 'opt_space')}, {32: 576}, ['element']), ({'T.whitespace': 86, "'title'": (32, 0, 'opt_space')}, {32: 577}, ['element']), ({'T.whitespace': 86, "'title'": (32, 0, 'opt_space')}, {32: 578}, ['element']), ({'T.whitespace': 86, "'script'": (32, 0, 'opt_space')}, {32: 579}, ['element']), ({'T.whitespace': 86, "'script'": (32, 0, 'opt_space')}, {32: 580}, ['element']), ({'T.whitespace': 86, "'li'": (32, 0, 'opt_space')}, {32: 581}, ['element']), ({'T.whitespace': 86, "'li'": (32, 0, 'opt_space')}, {32: 582}, ['element']), ({'T.whitespace': 86, "'dt'": (32, 0, 'opt_space')}, {32: 583}, ['element']), ({'T.whitespace': 86, "'dt'": (32, 0, 'opt_space')}, {32: 584}, ['element']), ({'T.whitespace': 86, "'dd'": (32, 0, 'opt_space')}, {32: 585}, ['element']), ({'T.whitespace': 86, "'dd'": (32, 0, 'opt_space')}, {32: 586}, ['element']), ({'T.whitespace': 86, "'ul'": (32, 0, 'opt_space')}, {32: 587}, ['element']), ({'T.whitespace': 86, "'ul'": (32, 0, 'opt_space')}, {32: 588}, ['element']), ({'T.whitespace': 86, "'ol'": (32, 0, 'opt_space')}, {32: 589}, ['element']), ({'T.whitespace': 86, "'ol'": (32, 0, 'opt_space')}, {32: 590}, ['element']), ({'T.whitespace': 86, "'dl'": (32, 0, 'opt_space')}, {32: 591}, ['element']), ({'T.whitespace': 86, "'dl'": (32, 0, 'opt_space')}, {32: 592}, ['element']), ({'T.whitespace': 86, "'div'": (32, 0, 'opt_space')}, {32: 593}, ['element']), ({'T.whitespace': 86, "'div'": (32, 0, 'opt_space')}, {32: 594}, ['element']), ({"'span'": (32, 0, 'opt_space'), 'T.whitespace': 86}, {32: 595}, ['element']), ({"'span'": (32, 0, 'opt_space'), 'T.whitespace': 86}, {32: 596}, ['element']), ({'T.whitespace': 86, "'td'": (32, 0, 'opt_space')}, {32: 597}, ['element']), ({'T.whitespace': 86, "'td'": (32, 0, 'opt_space')}, {32: 598}, ['element']), ({'T.whitespace': 86, "'colgroup'": (32, 0, 'opt_space')}, {32: 599}, ['element']), ({'T.whitespace': 86, "'colgroup'": (32, 0, 'opt_space')}, {32: 600}, ['element']), ({"'tr'": (32, 0, 'opt_space'), 'T.whitespace': 86}, {32: 601}, ['element']), ({"'tr'": (32, 0, 'opt_space'), 'T.whitespace': 86}, {32: 602}, ['element']), ({'T.whitespace': 86, "'tbody'": (32, 0, 'opt_space')}, {32: 603}, ['element']), ({'T.whitespace': 86, "'tbody'": (32, 0, 'opt_space')}, {32: 604}, ['element']), ({'T.whitespace': 86, "'th'": (32, 0, 'opt_space')}, {32: 605}, ['element']), ({'T.whitespace': 86, "'th'": (32, 0, 'opt_space')}, {32: 606}, ['element']), ({'T.whitespace': 86, "'thead'": (32, 0, 'opt_space')}, {32: 607}, ['element']), ({'T.whitespace': 86, "'thead'": (32, 0, 'opt_space')}, {32: 608}, ['element']), ({'T.whitespace': 86, "'form'": (32, 0, 'opt_space')}, {32: 609}, ['element']), ({'T.whitespace': 86, "'form'": (32, 0, 'opt_space')}, {32: 610}, ['element']), ({'T.whitespace': 86, "'table'": (32, 0, 'opt_space')}, {32: 611}, ['element']), ({'T.whitespace': 86, "'table'": (32, 0, 'opt_space')}, {32: 612}, ['element']), ({'T.whitespace': 86, "'iframe'": (32, 0, 'opt_space')}, {32: 613}, ['element']), ({'T.whitespace': 86, "'iframe'": (32, 0, 'opt_space')}, {32: 614}, ['element']), ({'T.whitespace': 86, "'frame'": (32, 0, 'opt_space')}, {32: 615}, ['element']), ({'T.whitespace': 86, "'frame'": (32, 0, 'opt_space')}, {32: 616}, ['element']), ({'T.whitespace': 86, "'head'": (32, 0, 'opt_space')}, {32: 617}, ['element']), ({'T.whitespace': 86, "'head'": (32, 0, 'opt_space')}, {32: 618}, ['element']), ({'T.whitespace': 86, "'body'": (32, 0, 'opt_space')}, {32: 619}, ['element']), ({'T.whitespace': 86, "'body'": (32, 0, 'opt_space')}, {32: 620}, ['element']), ({'T.whitespace': 86, "'html'": (32, 0, 'opt_space')}, {32: 621}, ['element']), ({'T.whitespace': 86, "'html'": (32, 0, 'opt_space')}, {32: 622}, ['element']), ({"'object'": 623}, {}, ['element']), ({"'object'": 624}, {}, ['element']), ({'T.whitespace': (141, 5, 'attr'), "'/>'": (141, 5, 'attr'), "'>'": (141, 5, 'attr'), "'>/'": (141, 5, 'attr')}, {}, ['attr']), ({'T.whitespace': (141, 5, 'attr'), "'/>'": (141, 5, 'attr'), "'>'": (141, 5, 'attr'), "'>/'": (141, 5, 'attr')}, {}, ['attr']), ({"'blockquote'": 625}, {}, ['element']), ({"'blockquote'": 626}, {}, ['element']), ({"'i'": 627}, {}, ['element']), ({"'i'": 628}, {}, ['element']), ({'T.tag_name': 629, 'T.string': 630}, {}, ['attr']), ({"'big'": 631}, {}, ['element']), ({"'big'": 632}, {}, ['element']), ({"'b'": 633}, {}, ['element']), ({"'b'": 634}, {}, ['element']), ({"'tt'": 635}, {}, ['element']), ({"'tt'": 636}, {}, ['element']), ({"'pre'": 637}, {}, ['element']), ({"'pre'": 638}, {}, ['element']), ({"'address'": 639}, {}, ['element']), ({"'address'": 640}, {}, ['element']), ({"'em'": 641}, {}, ['element']), ({"'em'": 642}, {}, ['element']), ({"'strong'": 643}, {}, ['element']), ({"'strong'": 644}, {}, ['element']), ({"'dfn'": 645}, {}, ['element']), ({"'dfn'": 646}, {}, ['element']), ({"'code'": 647}, {}, ['element']), ({"'code'": 648}, {}, ['element']), ({"'samp'": 649}, {}, ['element']), ({"'samp'": 650}, {}, ['element']), ({"'kbd'": 651}, {}, ['element']), ({"'kbd'": 652}, {}, ['element']), ({"'var'": 653}, {}, ['element']), ({"'var'": 654}, {}, ['element']), ({"'cite'": 655}, {}, ['element']), ({"'cite'": 656}, {}, ['element']), ({"'abbr'": 657}, {}, ['element']), ({"'abbr'": 658}, {}, ['element']), ({"'acronym'": 659}, {}, ['element']), ({"'acronym'": 660}, {}, ['element']), ({"'a'": 661}, {}, ['element']), ({"'a'": 662}, {}, ['element']), ({"'h1'": 663}, {}, ['element']), ({"'h1'": 664}, {}, ['element']), ({"'h2'": 665}, {}, ['element']), ({"'h2'": 666}, {}, ['element']), ({"'h3'": 667}, {}, ['element']), ({"'h3'": 668}, {}, ['element']), ({"'h4'": 669}, {}, ['element']), ({"'h4'": 670}, {}, ['element']), ({"'h5'": 671}, {}, ['element']), ({"'h5'": 672}, {}, ['element']), ({"'h6'": 673}, {}, ['element']), ({"'h6'": 674}, {}, ['element']), ({"'p'": 675}, {}, ['element']), ({"'p'": 676}, {}, ['element']), ({"'title'": 677}, {}, ['element']), ({"'title'": 678}, {}, ['element']), ({"'script'": 679}, {}, ['element']), ({"'script'": 680}, {}, ['element']), ({"'li'": 681}, {}, ['element']), ({"'li'": 682}, {}, ['element']), ({"'dt'": 683}, {}, ['element']), ({"'dt'": 684}, {}, ['element']), ({"'dd'": 685}, {}, ['element']), ({"'dd'": 686}, {}, ['element']), ({"'ul'": 687}, {}, ['element']), ({"'ul'": 688}, {}, ['element']), ({"'ol'": 689}, {}, ['element']), ({"'ol'": 690}, {}, ['element']), ({"'dl'": 691}, {}, ['element']), ({"'dl'": 692}, {}, ['element']), ({"'div'": 693}, {}, ['element']), ({"'div'": 694}, {}, ['element']), ({"'span'": 695}, {}, ['element']), ({"'span'": 696}, {}, ['element']), ({"'td'": 697}, {}, ['element']), ({"'td'": 698}, {}, ['element']), ({"'colgroup'": 699}, {}, ['element']), ({"'colgroup'": 700}, {}, ['element']), ({"'tr'": 701}, {}, ['element']), ({"'tr'": 702}, {}, ['element']), ({"'tbody'": 703}, {}, ['element']), ({"'tbody'": 704}, {}, ['element']), ({"'th'": 705}, {}, ['element']), ({"'th'": 706}, {}, ['element']), ({"'thead'": 707}, {}, ['element']), ({"'thead'": 708}, {}, ['element']), ({"'form'": 709}, {}, ['element']), ({"'form'": 710}, {}, ['element']), ({"'table'": 711}, {}, ['element']), ({"'table'": 712}, {}, ['element']), ({"'iframe'": 713}, {}, ['element']), ({"'iframe'": 714}, {}, ['element']), ({"'frame'": 715}, {}, ['element']), ({"'frame'": 716}, {}, ['element']), ({"'head'": 717}, {}, ['element']), ({"'head'": 718}, {}, ['element']), ({"'body'": 719}, {}, ['element']), ({"'body'": 720}, {}, ['element']), ({"'html'": 721}, {}, ['element']), ({"'html'": 722}, {}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 725}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 726}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 727}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 728}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 729}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 733}, ['element']), ({'T.whitespace': (141, 5, 'attr'), "'/>'": (141, 5, 'attr'), "'>'": (141, 5, 'attr'), "'>/'": (141, 5, 'attr')}, {}, ['attr']), ({'T.whitespace': (141, 5, 'attr'), "'/>'": (141, 5, 'attr'), "'>'": (141, 5, 'attr'), "'>/'": (141, 5, 'attr')}, {}, ['attr']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 734}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 735}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 736}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 737}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 738}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 739}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 740}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 741}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 742}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 743}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 744}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 745}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 746}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 747}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 748}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 749}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 750}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 751}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 752}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 753}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 754}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 755}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 756}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 757}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 758}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 759}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 760}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 761}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 762}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 763}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 764}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 767}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 768}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 769}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 770}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 771}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 772}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 773}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 774}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 775}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 776}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 777}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 778}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 779}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 780}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 781}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 782}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 783}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 784}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 785}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 786}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 787}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 788}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 789}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 790}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 791}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 792}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 793}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 794}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 795}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 796}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 797}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 798}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 799}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 800}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 801}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 802}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 803}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 804}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 805}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 806}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 807}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 808}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 809}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 810}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 811}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 812}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 813}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 814}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 815}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 816}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 817}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 818}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 819}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 820}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 821}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 822}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 823}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 824}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 825}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 826}, ['element']), ({'T.whitespace': 723, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 724, 34: 827}, ['element']), ({'T.tag_name': 730, "'/>'": (32, 1, 'opt_space'), "'>'": (32, 1, 'opt_space')}, {89: 731, 141: 732}, ['attrs', 'opt_space']), ({"'/>'": (34, 1, 'attrs'), "'>'": (34, 1, 'attrs')}, {}, ['attrs']), ({"'/>'": 829, "'>'": 828}, {}, ['element']), ({"'/>'": 831, "'>'": 830}, {}, ['element']), ({"'/>'": 833, "'>'": 832}, {}, ['element']), ({"'/>'": 835, "'>'": 834}, {}, ['element']), ({"'/>'": 837, "'>'": 836}, {}, ['element']), ({'T.whitespace': (141, 1, 'attr'), "'/>'": (141, 1, 'attr'), "'='": (32, 0, 'opt_space'), "'>'": (141, 1, 'attr')}, {32: 838}, ['attr']), ({'T.whitespace': 765, "'/>'": (32, 0, 'opt_space'), "'>'": (32, 0, 'opt_space')}, {32: 766}, ['attrs']), ({'T.whitespace': (89, 1), "'/>'": (89, 1), "'>'": (89, 1)}, {}, ['attrs']), ({"'/>'": 840, "'>'": 839}, {}, ['element']), ({"'/>'": 842, "'>'": 841}, {}, ['element']), ({"'/>'": 844, "'>'": 843}, {}, ['element']), ({"'/>'": 846, "'>'": 845}, {}, ['element']), ({"'/>'": 848, "'>'": 847}, {}, ['element']), ({"'/>'": 850, "'>'": 849}, {}, ['element']), ({"'/>'": 852, "'>'": 851}, {}, ['element']), ({"'/>'": 854, "'>'": 853}, {}, ['element']), ({"'/>'": 856, "'>'": 855}, {}, ['element']), ({"'/>'": 858, "'>'": 857}, {}, ['element']), ({"'/>'": 860, "'>'": 859}, {}, ['element']), ({"'/>'": 862, "'>'": 861}, {}, ['element']), ({"'/>'": 864, "'>'": 863}, {}, ['element']), ({"'/>'": 866, "'>'": 865}, {}, ['element']), ({"'/>'": 868, "'>'": 867}, {}, ['element']), ({"'/>'": 870, "'>'": 869}, {}, ['element']), ({"'/>'": 872, "'>'": 871}, {}, ['element']), ({"'/>'": 874, "'>'": 873}, {}, ['element']), ({"'/>'": 876, "'>'": 875}, {}, ['element']), ({"'/>'": 878, "'>'": 877}, {}, ['element']), ({"'/>'": 880, "'>'": 879}, {}, ['element']), ({"'/>'": 882, "'>'": 881}, {}, ['element']), ({"'/>'": 884, "'>'": 883}, {}, ['element']), ({"'/>'": 886, "'>'": 885}, {}, ['element']), ({"'/>'": 888, "'>'": 887}, {}, ['element']), ({"'/>'": 890, "'>'": 889}, {}, ['element']), ({"'/>'": 892, "'>'": 891}, {}, ['element']), ({"'/>'": 894, "'>'": 893}, {}, ['element']), ({"'/>'": 896, "'>'": 895}, {}, ['element']), ({"'/>'": 898, "'>'": 897}, {}, ['element']), ({"'/>'": 900, "'>'": 899}, {}, ['element']), ({"'/>'": 902, "'>'": 901}, {}, ['element']), ({'T.tag_name': 730, "'/>'": (32, 1, 'opt_space'), "'>'": (32, 1, 'opt_space')}, {141: 903}, ['attrs', 'opt_space']), ({"'/>'": (34, 3, 'attrs'), "'>'": (34, 3, 'attrs')}, {}, ['attrs']), ({"'/>'": 905, "'>'": 904}, {}, ['element']), ({"'/>'": 907, "'>'": 906}, {}, ['element']), ({"'/>'": 909, "'>'": 908}, {}, ['element']), ({"'/>'": 911, "'>'": 910}, {}, ['element']), ({"'/>'": 913, "'>'": 912}, {}, ['element']), ({"'/>'": 915, "'>'": 914}, {}, ['element']), ({"'/>'": 917, "'>'": 916}, {}, ['element']), ({"'/>'": 919, "'>'": 918}, {}, ['element']), ({"'/>'": 921, "'>'": 920}, {}, ['element']), ({"'/>'": 923, "'>'": 922}, {}, ['element']), ({"'/>'": 925, "'>'": 924}, {}, ['element']), ({"'/>'": 927, "'>'": 926}, {}, ['element']), ({"'/>'": 929, "'>'": 928}, {}, ['element']), ({"'/>'": 931, "'>'": 930}, {}, ['element']), ({"'/>'": 933, "'>'": 932}, {}, ['element']), ({"'/>'": 935, "'>'": 934}, {}, ['element']), ({"'/>'": 937, "'>'": 936}, {}, ['element']), ({"'/>'": 939, "'>'": 938}, {}, ['element']), ({"'/>'": 941, "'>'": 940}, {}, ['element']), ({"'/>'": 943, "'>'": 942}, {}, ['element']), ({"'/>'": 945, "'>'": 944}, {}, ['element']), ({"'/>'": 947, "'>'": 946}, {}, ['element']), ({"'/>'": 949, "'>'": 948}, {}, ['element']), ({"'/>'": 951, "'>'": 950}, {}, ['element']), ({"'/>'": 953, "'>'": 952}, {}, ['element']), ({"'/>'": 955, "'>'": 954}, {}, ['element']), ({"'/>'": 957, "'>'": 956}, {}, ['element']), ({"'/>'": 959, "'>'": 958}, {}, ['element']), ({"'/>'": 961, "'>'": 960}, {}, ['element']), ({"'/>'": 963, "'>'": 962}, {}, ['element']), ({"'/>'": 965, "'>'": 964}, {}, ['element']), ({"'/>'": 967, "'>'": 966}, {}, ['element']), ({"'/>'": 969, "'>'": 968}, {}, ['element']), ({"'/>'": 971, "'>'": 970}, {}, ['element']), ({"'/>'": 973, "'>'": 972}, {}, ['element']), ({"'/>'": 975, "'>'": 974}, {}, ['element']), ({"'/>'": 977, "'>'": 976}, {}, ['element']), ({"'/>'": 979, "'>'": 978}, {}, ['element']), ({"'/>'": 981, "'>'": 980}, {}, ['element']), ({"'/>'": 983, "'>'": 982}, {}, ['element']), ({"'/>'": 985, "'>'": 984}, {}, ['element']), ({"'/>'": 987, "'>'": 986}, {}, ['element']), ({"'/>'": 989, "'>'": 988}, {}, ['element']), ({"'/>'": 991, "'>'": 990}, {}, ['element']), ({"'/>'": 993, "'>'": 992}, {}, ['element']), ({"'/>'": 995, "'>'": 994}, {}, ['element']), ({"'/>'": 997, "'>'": 996}, {}, ['element']), ({"'/>'": 999, "'>'": 998}, {}, ['element']), ({"'/>'": 1001, "'>'": 1000}, {}, ['element']), ({"'/>'": 1003, "'>'": 1002}, {}, ['element']), ({"'/>'": 1005, "'>'": 1004}, {}, ['element']), ({"'/>'": 1007, "'>'": 1006}, {}, ['element']), ({"'/>'": 1009, "'>'": 1008}, {}, ['element']), ({"'/>'": 1011, "'>'": 1010}, {}, ['element']), ({"'/>'": 1013, "'>'": 1012}, {}, ['element']), ({"'/>'": 1015, "'>'": 1014}, {}, ['element']), ({"'/>'": 1017, "'>'": 1016}, {}, ['element']), ({"'/>'": 1019, "'>'": 1018}, {}, ['element']), ({"'/>'": 1021, "'>'": 1020}, {}, ['element']), ({"'/>'": 1023, "'>'": 1022}, {}, ['element']), ({"'/>'": 1025, "'>'": 1024}, {}, ['element']), ({'T.text': (38, 11), '__end_of_input__': (38, 11), 'T.whitespace': (38, 11), "'<!'": (38, 11), 'T.text_end': (38, 11), "'</'": (38, 11), "'<'": (38, 11)}, {}, ['element']), ({'T.text': (38, 11), '__end_of_input__': (38, 11), 'T.whitespace': (38, 11), "'<!'": (38, 11), 'T.text_end': (38, 11), "'</'": (38, 11), "'<'": (38, 11)}, {}, ['element']), ({'T.text': (38, 11), '__end_of_input__': (38, 11), 'T.whitespace': (38, 11), "'<!'": (38, 11), 'T.text_end': (38, 11), "'</'": (38, 11), "'<'": (38, 11)}, {}, ['element']), ({'T.text': (38, 11), '__end_of_input__': (38, 11), 'T.whitespace': (38, 11), "'<!'": (38, 11), 'T.text_end': (38, 11), "'</'": (38, 11), "'<'": (38, 11)}, {}, ['element']), ({'T.text': (39, 11), '__end_of_input__': (39, 11), 'T.whitespace': (39, 11), "'<!'": (39, 11), 'T.text_end': (39, 11), "'</'": (39, 11), "'<'": (39, 11)}, {}, ['element']), ({'T.text': (39, 11), '__end_of_input__': (39, 11), 'T.whitespace': (39, 11), "'<!'": (39, 11), 'T.text_end': (39, 11), "'</'": (39, 11), "'<'": (39, 11)}, {}, ['element']), ({'T.text': (39, 11), '__end_of_input__': (39, 11), 'T.whitespace': (39, 11), "'<!'": (39, 11), 'T.text_end': (39, 11), "'</'": (39, 11), "'<'": (39, 11)}, {}, ['element']), ({'T.text': (39, 11), '__end_of_input__': (39, 11), 'T.whitespace': (39, 11), "'<!'": (39, 11), 'T.text_end': (39, 11), "'</'": (39, 11), "'<'": (39, 11)}, {}, ['element']), ({'T.text': (40, 11), '__end_of_input__': (40, 11), 'T.whitespace': (40, 11), "'<!'": (40, 11), 'T.text_end': (40, 11), "'</'": (40, 11), "'<'": (40, 11)}, {}, ['element']), ({'T.text': (40, 11), '__end_of_input__': (40, 11), 'T.whitespace': (40, 11), "'<!'": (40, 11), 'T.text_end': (40, 11), "'</'": (40, 11), "'<'": (40, 11)}, {}, ['element']), ({"'='": 1026}, {}, ['attr']), ({'T.text': (40, 11), '__end_of_input__': (40, 11), 'T.whitespace': (40, 11), "'<!'": (40, 11), 'T.text_end': (40, 11), "'</'": (40, 11), "'<'": (40, 11)}, {}, ['element']), ({'T.text': (40, 11), '__end_of_input__': (40, 11), 'T.whitespace': (40, 11), "'<!'": (40, 11), 'T.text_end': (40, 11), "'</'": (40, 11), "'<'": (40, 11)}, {}, ['element']), ({'T.text': (41, 11), '__end_of_input__': (41, 11), 'T.whitespace': (41, 11), "'<!'": (41, 11), 'T.text_end': (41, 11), "'</'": (41, 11), "'<'": (41, 11)}, {}, ['element']), ({'T.text': (41, 11), '__end_of_input__': (41, 11), 'T.whitespace': (41, 11), "'<!'": (41, 11), 'T.text_end': (41, 11), "'</'": (41, 11), "'<'": (41, 11)}, {}, ['element']), ({'T.text': (41, 11), '__end_of_input__': (41, 11), 'T.whitespace': (41, 11), "'<!'": (41, 11), 'T.text_end': (41, 11), "'</'": (41, 11), "'<'": (41, 11)}, {}, ['element']), ({'T.text': (41, 11), '__end_of_input__': (41, 11), 'T.whitespace': (41, 11), "'<!'": (41, 11), 'T.text_end': (41, 11), "'</'": (41, 11), "'<'": (41, 11)}, {}, ['element']), ({'T.text': (42, 11), '__end_of_input__': (42, 11), 'T.whitespace': (42, 11), "'<!'": (42, 11), 'T.text_end': (42, 11), "'</'": (42, 11), "'<'": (42, 11)}, {}, ['element']), ({'T.text': (42, 11), '__end_of_input__': (42, 11), 'T.whitespace': (42, 11), "'<!'": (42, 11), 'T.text_end': (42, 11), "'</'": (42, 11), "'<'": (42, 11)}, {}, ['element']), ({'T.text': (42, 11), '__end_of_input__': (42, 11), 'T.whitespace': (42, 11), "'<!'": (42, 11), 'T.text_end': (42, 11), "'</'": (42, 11), "'<'": (42, 11)}, {}, ['element']), ({'T.text': (42, 11), '__end_of_input__': (42, 11), 'T.whitespace': (42, 11), "'<!'": (42, 11), 'T.text_end': (42, 11), "'</'": (42, 11), "'<'": (42, 11)}, {}, ['element']), ({'T.text': (43, 11), '__end_of_input__': (43, 11), 'T.whitespace': (43, 11), "'<!'": (43, 11), 'T.text_end': (43, 11), "'</'": (43, 11), "'<'": (43, 11)}, {}, ['element']), ({'T.text': (43, 11), '__end_of_input__': (43, 11), 'T.whitespace': (43, 11), "'<!'": (43, 11), 'T.text_end': (43, 11), "'</'": (43, 11), "'<'": (43, 11)}, {}, ['element']), ({'T.text': (43, 11), '__end_of_input__': (43, 11), 'T.whitespace': (43, 11), "'<!'": (43, 11), 'T.text_end': (43, 11), "'</'": (43, 11), "'<'": (43, 11)}, {}, ['element']), ({'T.text': (43, 11), '__end_of_input__': (43, 11), 'T.whitespace': (43, 11), "'<!'": (43, 11), 'T.text_end': (43, 11), "'</'": (43, 11), "'<'": (43, 11)}, {}, ['element']), ({'T.text': (44, 11), '__end_of_input__': (44, 11), 'T.whitespace': (44, 11), "'<!'": (44, 11), 'T.text_end': (44, 11), "'</'": (44, 11), "'<'": (44, 11)}, {}, ['element']), ({'T.text': (44, 11), '__end_of_input__': (44, 11), 'T.whitespace': (44, 11), "'<!'": (44, 11), 'T.text_end': (44, 11), "'</'": (44, 11), "'<'": (44, 11)}, {}, ['element']), ({'T.text': (44, 11), '__end_of_input__': (44, 11), 'T.whitespace': (44, 11), "'<!'": (44, 11), 'T.text_end': (44, 11), "'</'": (44, 11), "'<'": (44, 11)}, {}, ['element']), ({'T.text': (44, 11), '__end_of_input__': (44, 11), 'T.whitespace': (44, 11), "'<!'": (44, 11), 'T.text_end': (44, 11), "'</'": (44, 11), "'<'": (44, 11)}, {}, ['element']), ({'T.text': (45, 11), '__end_of_input__': (45, 11), 'T.whitespace': (45, 11), "'<!'": (45, 11), 'T.text_end': (45, 11), "'</'": (45, 11), "'<'": (45, 11)}, {}, ['element']), ({'T.text': (45, 11), '__end_of_input__': (45, 11), 'T.whitespace': (45, 11), "'<!'": (45, 11), 'T.text_end': (45, 11), "'</'": (45, 11), "'<'": (45, 11)}, {}, ['element']), ({'T.text': (45, 11), '__end_of_input__': (45, 11), 'T.whitespace': (45, 11), "'<!'": (45, 11), 'T.text_end': (45, 11), "'</'": (45, 11), "'<'": (45, 11)}, {}, ['element']), ({'T.text': (45, 11), '__end_of_input__': (45, 11), 'T.whitespace': (45, 11), "'<!'": (45, 11), 'T.text_end': (45, 11), "'</'": (45, 11), "'<'": (45, 11)}, {}, ['element']), ({'T.text': (46, 11), '__end_of_input__': (46, 11), 'T.whitespace': (46, 11), "'<!'": (46, 11), 'T.text_end': (46, 11), "'</'": (46, 11), "'<'": (46, 11)}, {}, ['element']), ({'T.text': (46, 11), '__end_of_input__': (46, 11), 'T.whitespace': (46, 11), "'<!'": (46, 11), 'T.text_end': (46, 11), "'</'": (46, 11), "'<'": (46, 11)}, {}, ['element']), ({'T.text': (46, 11), '__end_of_input__': (46, 11), 'T.whitespace': (46, 11), "'<!'": (46, 11), 'T.text_end': (46, 11), "'</'": (46, 11), "'<'": (46, 11)}, {}, ['element']), ({'T.text': (46, 11), '__end_of_input__': (46, 11), 'T.whitespace': (46, 11), "'<!'": (46, 11), 'T.text_end': (46, 11), "'</'": (46, 11), "'<'": (46, 11)}, {}, ['element']), ({'T.text': (47, 11), '__end_of_input__': (47, 11), 'T.whitespace': (47, 11), "'<!'": (47, 11), 'T.text_end': (47, 11), "'</'": (47, 11), "'<'": (47, 11)}, {}, ['element']), ({'T.text': (47, 11), '__end_of_input__': (47, 11), 'T.whitespace': (47, 11), "'<!'": (47, 11), 'T.text_end': (47, 11), "'</'": (47, 11), "'<'": (47, 11)}, {}, ['element']), ({'T.text': (47, 11), '__end_of_input__': (47, 11), 'T.whitespace': (47, 11), "'<!'": (47, 11), 'T.text_end': (47, 11), "'</'": (47, 11), "'<'": (47, 11)}, {}, ['element']), ({'T.text': (47, 11), '__end_of_input__': (47, 11), 'T.whitespace': (47, 11), "'<!'": (47, 11), 'T.text_end': (47, 11), "'</'": (47, 11), "'<'": (47, 11)}, {}, ['element']), ({'T.text': (48, 11), '__end_of_input__': (48, 11), 'T.whitespace': (48, 11), "'<!'": (48, 11), 'T.text_end': (48, 11), "'</'": (48, 11), "'<'": (48, 11)}, {}, ['element']), ({'T.text': (48, 11), '__end_of_input__': (48, 11), 'T.whitespace': (48, 11), "'<!'": (48, 11), 'T.text_end': (48, 11), "'</'": (48, 11), "'<'": (48, 11)}, {}, ['element']), ({'T.text': (48, 11), '__end_of_input__': (48, 11), 'T.whitespace': (48, 11), "'<!'": (48, 11), 'T.text_end': (48, 11), "'</'": (48, 11), "'<'": (48, 11)}, {}, ['element']), ({'T.text': (48, 11), '__end_of_input__': (48, 11), 'T.whitespace': (48, 11), "'<!'": (48, 11), 'T.text_end': (48, 11), "'</'": (48, 11), "'<'": (48, 11)}, {}, ['element']), ({'T.text': (49, 11), '__end_of_input__': (49, 11), 'T.whitespace': (49, 11), "'<!'": (49, 11), 'T.text_end': (49, 11), "'</'": (49, 11), "'<'": (49, 11)}, {}, ['element']), ({'T.text': (49, 11), '__end_of_input__': (49, 11), 'T.whitespace': (49, 11), "'<!'": (49, 11), 'T.text_end': (49, 11), "'</'": (49, 11), "'<'": (49, 11)}, {}, ['element']), ({'T.text': (49, 11), '__end_of_input__': (49, 11), 'T.whitespace': (49, 11), "'<!'": (49, 11), 'T.text_end': (49, 11), "'</'": (49, 11), "'<'": (49, 11)}, {}, ['element']), ({'T.text': (49, 11), '__end_of_input__': (49, 11), 'T.whitespace': (49, 11), "'<!'": (49, 11), 'T.text_end': (49, 11), "'</'": (49, 11), "'<'": (49, 11)}, {}, ['element']), ({'T.text': (50, 11), '__end_of_input__': (50, 11), 'T.whitespace': (50, 11), "'<!'": (50, 11), 'T.text_end': (50, 11), "'</'": (50, 11), "'<'": (50, 11)}, {}, ['element']), ({'T.text': (50, 11), '__end_of_input__': (50, 11), 'T.whitespace': (50, 11), "'<!'": (50, 11), 'T.text_end': (50, 11), "'</'": (50, 11), "'<'": (50, 11)}, {}, ['element']), ({'T.text': (50, 11), '__end_of_input__': (50, 11), 'T.whitespace': (50, 11), "'<!'": (50, 11), 'T.text_end': (50, 11), "'</'": (50, 11), "'<'": (50, 11)}, {}, ['element']), ({'T.text': (50, 11), '__end_of_input__': (50, 11), 'T.whitespace': (50, 11), "'<!'": (50, 11), 'T.text_end': (50, 11), "'</'": (50, 11), "'<'": (50, 11)}, {}, ['element']), ({'T.text': (51, 11), '__end_of_input__': (51, 11), 'T.whitespace': (51, 11), "'<!'": (51, 11), 'T.text_end': (51, 11), "'</'": (51, 11), "'<'": (51, 11)}, {}, ['element']), ({'T.text': (51, 11), '__end_of_input__': (51, 11), 'T.whitespace': (51, 11), "'<!'": (51, 11), 'T.text_end': (51, 11), "'</'": (51, 11), "'<'": (51, 11)}, {}, ['element']), ({'T.text': (51, 11), '__end_of_input__': (51, 11), 'T.whitespace': (51, 11), "'<!'": (51, 11), 'T.text_end': (51, 11), "'</'": (51, 11), "'<'": (51, 11)}, {}, ['element']), ({'T.text': (51, 11), '__end_of_input__': (51, 11), 'T.whitespace': (51, 11), "'<!'": (51, 11), 'T.text_end': (51, 11), "'</'": (51, 11), "'<'": (51, 11)}, {}, ['element']), ({'T.text': (52, 11), '__end_of_input__': (52, 11), 'T.whitespace': (52, 11), "'<!'": (52, 11), 'T.text_end': (52, 11), "'</'": (52, 11), "'<'": (52, 11)}, {}, ['element']), ({'T.text': (52, 11), '__end_of_input__': (52, 11), 'T.whitespace': (52, 11), "'<!'": (52, 11), 'T.text_end': (52, 11), "'</'": (52, 11), "'<'": (52, 11)}, {}, ['element']), ({'T.text': (52, 11), '__end_of_input__': (52, 11), 'T.whitespace': (52, 11), "'<!'": (52, 11), 'T.text_end': (52, 11), "'</'": (52, 11), "'<'": (52, 11)}, {}, ['element']), ({'T.text': (52, 11), '__end_of_input__': (52, 11), 'T.whitespace': (52, 11), "'<!'": (52, 11), 'T.text_end': (52, 11), "'</'": (52, 11), "'<'": (52, 11)}, {}, ['element']), ({'T.text': (53, 11), '__end_of_input__': (53, 11), 'T.whitespace': (53, 11), "'<!'": (53, 11), 'T.text_end': (53, 11), "'</'": (53, 11), "'<'": (53, 11)}, {}, ['element']), ({'T.text': (53, 11), '__end_of_input__': (53, 11), 'T.whitespace': (53, 11), "'<!'": (53, 11), 'T.text_end': (53, 11), "'</'": (53, 11), "'<'": (53, 11)}, {}, ['element']), ({'T.text': (53, 11), '__end_of_input__': (53, 11), 'T.whitespace': (53, 11), "'<!'": (53, 11), 'T.text_end': (53, 11), "'</'": (53, 11), "'<'": (53, 11)}, {}, ['element']), ({'T.text': (53, 11), '__end_of_input__': (53, 11), 'T.whitespace': (53, 11), "'<!'": (53, 11), 'T.text_end': (53, 11), "'</'": (53, 11), "'<'": (53, 11)}, {}, ['element']), ({'T.text': (54, 11), '__end_of_input__': (54, 11), 'T.whitespace': (54, 11), "'<!'": (54, 11), 'T.text_end': (54, 11), "'</'": (54, 11), "'<'": (54, 11)}, {}, ['element']), ({'T.text': (54, 11), '__end_of_input__': (54, 11), 'T.whitespace': (54, 11), "'<!'": (54, 11), 'T.text_end': (54, 11), "'</'": (54, 11), "'<'": (54, 11)}, {}, ['element']), ({'T.text': (54, 11), '__end_of_input__': (54, 11), 'T.whitespace': (54, 11), "'<!'": (54, 11), 'T.text_end': (54, 11), "'</'": (54, 11), "'<'": (54, 11)}, {}, ['element']), ({'T.text': (54, 11), '__end_of_input__': (54, 11), 'T.whitespace': (54, 11), "'<!'": (54, 11), 'T.text_end': (54, 11), "'</'": (54, 11), "'<'": (54, 11)}, {}, ['element']), ({'T.text': (55, 11), '__end_of_input__': (55, 11), 'T.whitespace': (55, 11), "'<!'": (55, 11), 'T.text_end': (55, 11), "'</'": (55, 11), "'<'": (55, 11)}, {}, ['element']), ({'T.text': (55, 11), '__end_of_input__': (55, 11), 'T.whitespace': (55, 11), "'<!'": (55, 11), 'T.text_end': (55, 11), "'</'": (55, 11), "'<'": (55, 11)}, {}, ['element']), ({'T.text': (55, 11), '__end_of_input__': (55, 11), 'T.whitespace': (55, 11), "'<!'": (55, 11), 'T.text_end': (55, 11), "'</'": (55, 11), "'<'": (55, 11)}, {}, ['element']), ({'T.text': (55, 11), '__end_of_input__': (55, 11), 'T.whitespace': (55, 11), "'<!'": (55, 11), 'T.text_end': (55, 11), "'</'": (55, 11), "'<'": (55, 11)}, {}, ['element']), ({'T.text': (56, 11), '__end_of_input__': (56, 11), 'T.whitespace': (56, 11), "'<!'": (56, 11), 'T.text_end': (56, 11), "'</'": (56, 11), "'<'": (56, 11)}, {}, ['element']), ({'T.text': (56, 11), '__end_of_input__': (56, 11), 'T.whitespace': (56, 11), "'<!'": (56, 11), 'T.text_end': (56, 11), "'</'": (56, 11), "'<'": (56, 11)}, {}, ['element']), ({'T.whitespace': (89, 3), "'/>'": (89, 3), "'>'": (89, 3)}, {}, ['attrs']), ({'T.text': (56, 11), '__end_of_input__': (56, 11), 'T.whitespace': (56, 11), "'<!'": (56, 11), 'T.text_end': (56, 11), "'</'": (56, 11), "'<'": (56, 11)}, {}, ['element']), ({'T.text': (56, 11), '__end_of_input__': (56, 11), 'T.whitespace': (56, 11), "'<!'": (56, 11), 'T.text_end': (56, 11), "'</'": (56, 11), "'<'": (56, 11)}, {}, ['element']), ({'T.text': (57, 11), '__end_of_input__': (57, 11), 'T.whitespace': (57, 11), "'<!'": (57, 11), 'T.text_end': (57, 11), "'</'": (57, 11), "'<'": (57, 11)}, {}, ['element']), ({'T.text': (57, 11), '__end_of_input__': (57, 11), 'T.whitespace': (57, 11), "'<!'": (57, 11), 'T.text_end': (57, 11), "'</'": (57, 11), "'<'": (57, 11)}, {}, ['element']), ({'T.text': (57, 11), '__end_of_input__': (57, 11), 'T.whitespace': (57, 11), "'<!'": (57, 11), 'T.text_end': (57, 11), "'</'": (57, 11), "'<'": (57, 11)}, {}, ['element']), ({'T.text': (57, 11), '__end_of_input__': (57, 11), 'T.whitespace': (57, 11), "'<!'": (57, 11), 'T.text_end': (57, 11), "'</'": (57, 11), "'<'": (57, 11)}, {}, ['element']), ({'T.text': (58, 11), '__end_of_input__': (58, 11), 'T.whitespace': (58, 11), "'<!'": (58, 11), 'T.text_end': (58, 11), "'</'": (58, 11), "'<'": (58, 11)}, {}, ['element']), ({'T.text': (58, 11), '__end_of_input__': (58, 11), 'T.whitespace': (58, 11), "'<!'": (58, 11), 'T.text_end': (58, 11), "'</'": (58, 11), "'<'": (58, 11)}, {}, ['element']), ({'T.text': (58, 11), '__end_of_input__': (58, 11), 'T.whitespace': (58, 11), "'<!'": (58, 11), 'T.text_end': (58, 11), "'</'": (58, 11), "'<'": (58, 11)}, {}, ['element']), ({'T.text': (58, 11), '__end_of_input__': (58, 11), 'T.whitespace': (58, 11), "'<!'": (58, 11), 'T.text_end': (58, 11), "'</'": (58, 11), "'<'": (58, 11)}, {}, ['element']), ({'T.text': (59, 11), '__end_of_input__': (59, 11), 'T.whitespace': (59, 11), "'<!'": (59, 11), 'T.text_end': (59, 11), "'</'": (59, 11), "'<'": (59, 11)}, {}, ['element']), ({'T.text': (59, 11), '__end_of_input__': (59, 11), 'T.whitespace': (59, 11), "'<!'": (59, 11), 'T.text_end': (59, 11), "'</'": (59, 11), "'<'": (59, 11)}, {}, ['element']), ({'T.text': (59, 11), '__end_of_input__': (59, 11), 'T.whitespace': (59, 11), "'<!'": (59, 11), 'T.text_end': (59, 11), "'</'": (59, 11), "'<'": (59, 11)}, {}, ['element']), ({'T.text': (59, 11), '__end_of_input__': (59, 11), 'T.whitespace': (59, 11), "'<!'": (59, 11), 'T.text_end': (59, 11), "'</'": (59, 11), "'<'": (59, 11)}, {}, ['element']), ({'T.text': (60, 11), '__end_of_input__': (60, 11), 'T.whitespace': (60, 11), "'<!'": (60, 11), 'T.text_end': (60, 11), "'</'": (60, 11), "'<'": (60, 11)}, {}, ['element']), ({'T.text': (60, 11), '__end_of_input__': (60, 11), 'T.whitespace': (60, 11), "'<!'": (60, 11), 'T.text_end': (60, 11), "'</'": (60, 11), "'<'": (60, 11)}, {}, ['element']), ({'T.text': (60, 11), '__end_of_input__': (60, 11), 'T.whitespace': (60, 11), "'<!'": (60, 11), 'T.text_end': (60, 11), "'</'": (60, 11), "'<'": (60, 11)}, {}, ['element']), ({'T.text': (60, 11), '__end_of_input__': (60, 11), 'T.whitespace': (60, 11), "'<!'": (60, 11), 'T.text_end': (60, 11), "'</'": (60, 11), "'<'": (60, 11)}, {}, ['element']), ({'T.text': (61, 11), '__end_of_input__': (61, 11), 'T.whitespace': (61, 11), "'<!'": (61, 11), 'T.text_end': (61, 11), "'</'": (61, 11), "'<'": (61, 11)}, {}, ['element']), ({'T.text': (61, 11), '__end_of_input__': (61, 11), 'T.whitespace': (61, 11), "'<!'": (61, 11), 'T.text_end': (61, 11), "'</'": (61, 11), "'<'": (61, 11)}, {}, ['element']), ({'T.text': (61, 11), '__end_of_input__': (61, 11), 'T.whitespace': (61, 11), "'<!'": (61, 11), 'T.text_end': (61, 11), "'</'": (61, 11), "'<'": (61, 11)}, {}, ['element']), ({'T.text': (61, 11), '__end_of_input__': (61, 11), 'T.whitespace': (61, 11), "'<!'": (61, 11), 'T.text_end': (61, 11), "'</'": (61, 11), "'<'": (61, 11)}, {}, ['element']), ({'T.text': (62, 11), '__end_of_input__': (62, 11), 'T.whitespace': (62, 11), "'<!'": (62, 11), 'T.text_end': (62, 11), "'</'": (62, 11), "'<'": (62, 11)}, {}, ['element']), ({'T.text': (62, 11), '__end_of_input__': (62, 11), 'T.whitespace': (62, 11), "'<!'": (62, 11), 'T.text_end': (62, 11), "'</'": (62, 11), "'<'": (62, 11)}, {}, ['element']), ({'T.text': (62, 11), '__end_of_input__': (62, 11), 'T.whitespace': (62, 11), "'<!'": (62, 11), 'T.text_end': (62, 11), "'</'": (62, 11), "'<'": (62, 11)}, {}, ['element']), ({'T.text': (62, 11), '__end_of_input__': (62, 11), 'T.whitespace': (62, 11), "'<!'": (62, 11), 'T.text_end': (62, 11), "'</'": (62, 11), "'<'": (62, 11)}, {}, ['element']), ({'T.text': (63, 11), '__end_of_input__': (63, 11), 'T.whitespace': (63, 11), "'<!'": (63, 11), 'T.text_end': (63, 11), "'</'": (63, 11), "'<'": (63, 11)}, {}, ['element']), ({'T.text': (63, 11), '__end_of_input__': (63, 11), 'T.whitespace': (63, 11), "'<!'": (63, 11), 'T.text_end': (63, 11), "'</'": (63, 11), "'<'": (63, 11)}, {}, ['element']), ({'T.text': (63, 11), '__end_of_input__': (63, 11), 'T.whitespace': (63, 11), "'<!'": (63, 11), 'T.text_end': (63, 11), "'</'": (63, 11), "'<'": (63, 11)}, {}, ['element']), ({'T.text': (63, 11), '__end_of_input__': (63, 11), 'T.whitespace': (63, 11), "'<!'": (63, 11), 'T.text_end': (63, 11), "'</'": (63, 11), "'<'": (63, 11)}, {}, ['element']), ({'T.text': (64, 11), '__end_of_input__': (64, 11), 'T.whitespace': (64, 11), "'<!'": (64, 11), 'T.text_end': (64, 11), "'</'": (64, 11), "'<'": (64, 11)}, {}, ['element']), ({'T.text': (64, 11), '__end_of_input__': (64, 11), 'T.whitespace': (64, 11), "'<!'": (64, 11), 'T.text_end': (64, 11), "'</'": (64, 11), "'<'": (64, 11)}, {}, ['element']), ({'T.text': (64, 11), '__end_of_input__': (64, 11), 'T.whitespace': (64, 11), "'<!'": (64, 11), 'T.text_end': (64, 11), "'</'": (64, 11), "'<'": (64, 11)}, {}, ['element']), ({'T.text': (64, 11), '__end_of_input__': (64, 11), 'T.whitespace': (64, 11), "'<!'": (64, 11), 'T.text_end': (64, 11), "'</'": (64, 11), "'<'": (64, 11)}, {}, ['element']), ({'T.text': (65, 11), '__end_of_input__': (65, 11), 'T.whitespace': (65, 11), "'<!'": (65, 11), 'T.text_end': (65, 11), "'</'": (65, 11), "'<'": (65, 11)}, {}, ['element']), ({'T.text': (65, 11), '__end_of_input__': (65, 11), 'T.whitespace': (65, 11), "'<!'": (65, 11), 'T.text_end': (65, 11), "'</'": (65, 11), "'<'": (65, 11)}, {}, ['element']), ({'T.text': (65, 11), '__end_of_input__': (65, 11), 'T.whitespace': (65, 11), "'<!'": (65, 11), 'T.text_end': (65, 11), "'</'": (65, 11), "'<'": (65, 11)}, {}, ['element']), ({'T.text': (65, 11), '__end_of_input__': (65, 11), 'T.whitespace': (65, 11), "'<!'": (65, 11), 'T.text_end': (65, 11), "'</'": (65, 11), "'<'": (65, 11)}, {}, ['element']), ({'T.text': (66, 11), '__end_of_input__': (66, 11), 'T.whitespace': (66, 11), "'<!'": (66, 11), 'T.text_end': (66, 11), "'</'": (66, 11), "'<'": (66, 11)}, {}, ['element']), ({'T.text': (66, 11), '__end_of_input__': (66, 11), 'T.whitespace': (66, 11), "'<!'": (66, 11), 'T.text_end': (66, 11), "'</'": (66, 11), "'<'": (66, 11)}, {}, ['element']), ({'T.text': (66, 11), '__end_of_input__': (66, 11), 'T.whitespace': (66, 11), "'<!'": (66, 11), 'T.text_end': (66, 11), "'</'": (66, 11), "'<'": (66, 11)}, {}, ['element']), ({'T.text': (66, 11), '__end_of_input__': (66, 11), 'T.whitespace': (66, 11), "'<!'": (66, 11), 'T.text_end': (66, 11), "'</'": (66, 11), "'<'": (66, 11)}, {}, ['element']), ({'T.text': (67, 11), '__end_of_input__': (67, 11), 'T.whitespace': (67, 11), "'<!'": (67, 11), 'T.text_end': (67, 11), "'</'": (67, 11), "'<'": (67, 11)}, {}, ['element']), ({'T.text': (67, 11), '__end_of_input__': (67, 11), 'T.whitespace': (67, 11), "'<!'": (67, 11), 'T.text_end': (67, 11), "'</'": (67, 11), "'<'": (67, 11)}, {}, ['element']), ({'T.text': (67, 11), '__end_of_input__': (67, 11), 'T.whitespace': (67, 11), "'<!'": (67, 11), 'T.text_end': (67, 11), "'</'": (67, 11), "'<'": (67, 11)}, {}, ['element']), ({'T.text': (67, 11), '__end_of_input__': (67, 11), 'T.whitespace': (67, 11), "'<!'": (67, 11), 'T.text_end': (67, 11), "'</'": (67, 11), "'<'": (67, 11)}, {}, ['element']), ({'T.text': (68, 11), '__end_of_input__': (68, 11), 'T.whitespace': (68, 11), "'<!'": (68, 11), 'T.text_end': (68, 11), "'</'": (68, 11), "'<'": (68, 11)}, {}, ['element']), ({'T.text': (68, 11), '__end_of_input__': (68, 11), 'T.whitespace': (68, 11), "'<!'": (68, 11), 'T.text_end': (68, 11), "'</'": (68, 11), "'<'": (68, 11)}, {}, ['element']), ({'T.text': (68, 11), '__end_of_input__': (68, 11), 'T.whitespace': (68, 11), "'<!'": (68, 11), 'T.text_end': (68, 11), "'</'": (68, 11), "'<'": (68, 11)}, {}, ['element']), ({'T.text': (68, 11), '__end_of_input__': (68, 11), 'T.whitespace': (68, 11), "'<!'": (68, 11), 'T.text_end': (68, 11), "'</'": (68, 11), "'<'": (68, 11)}, {}, ['element']), ({'T.text': (69, 11), '__end_of_input__': (69, 11), 'T.whitespace': (69, 11), "'<!'": (69, 11), 'T.text_end': (69, 11), "'</'": (69, 11), "'<'": (69, 11)}, {}, ['element']), ({'T.text': (69, 11), '__end_of_input__': (69, 11), 'T.whitespace': (69, 11), "'<!'": (69, 11), 'T.text_end': (69, 11), "'</'": (69, 11), "'<'": (69, 11)}, {}, ['element']), ({'T.text': (69, 11), '__end_of_input__': (69, 11), 'T.whitespace': (69, 11), "'<!'": (69, 11), 'T.text_end': (69, 11), "'</'": (69, 11), "'<'": (69, 11)}, {}, ['element']), ({'T.text': (69, 11), '__end_of_input__': (69, 11), 'T.whitespace': (69, 11), "'<!'": (69, 11), 'T.text_end': (69, 11), "'</'": (69, 11), "'<'": (69, 11)}, {}, ['element']), ({'T.text': (70, 11), '__end_of_input__': (70, 11), 'T.whitespace': (70, 11), "'<!'": (70, 11), 'T.text_end': (70, 11), "'</'": (70, 11), "'<'": (70, 11)}, {}, ['element']), ({'T.text': (70, 11), '__end_of_input__': (70, 11), 'T.whitespace': (70, 11), "'<!'": (70, 11), 'T.text_end': (70, 11), "'</'": (70, 11), "'<'": (70, 11)}, {}, ['element']), ({'T.text': (70, 11), '__end_of_input__': (70, 11), 'T.whitespace': (70, 11), "'<!'": (70, 11), 'T.text_end': (70, 11), "'</'": (70, 11), "'<'": (70, 11)}, {}, ['element']), ({'T.text': (70, 11), '__end_of_input__': (70, 11), 'T.whitespace': (70, 11), "'<!'": (70, 11), 'T.text_end': (70, 11), "'</'": (70, 11), "'<'": (70, 11)}, {}, ['element']), ({'T.text': (71, 11), '__end_of_input__': (71, 11), 'T.whitespace': (71, 11), "'<!'": (71, 11), 'T.text_end': (71, 11), "'</'": (71, 11), "'<'": (71, 11)}, {}, ['element']), ({'T.text': (71, 11), '__end_of_input__': (71, 11), 'T.whitespace': (71, 11), "'<!'": (71, 11), 'T.text_end': (71, 11), "'</'": (71, 11), "'<'": (71, 11)}, {}, ['element']), ({'T.text': (71, 11), '__end_of_input__': (71, 11), 'T.whitespace': (71, 11), "'<!'": (71, 11), 'T.text_end': (71, 11), "'</'": (71, 11), "'<'": (71, 11)}, {}, ['element']), ({'T.text': (71, 11), '__end_of_input__': (71, 11), 'T.whitespace': (71, 11), "'<!'": (71, 11), 'T.text_end': (71, 11), "'</'": (71, 11), "'<'": (71, 11)}, {}, ['element']), ({'T.text': (72, 11), '__end_of_input__': (72, 11), 'T.whitespace': (72, 11), "'<!'": (72, 11), 'T.text_end': (72, 11), "'</'": (72, 11), "'<'": (72, 11)}, {}, ['element']), ({'T.text': (72, 11), '__end_of_input__': (72, 11), 'T.whitespace': (72, 11), "'<!'": (72, 11), 'T.text_end': (72, 11), "'</'": (72, 11), "'<'": (72, 11)}, {}, ['element']), ({'T.text': (72, 11), '__end_of_input__': (72, 11), 'T.whitespace': (72, 11), "'<!'": (72, 11), 'T.text_end': (72, 11), "'</'": (72, 11), "'<'": (72, 11)}, {}, ['element']), ({'T.text': (72, 11), '__end_of_input__': (72, 11), 'T.whitespace': (72, 11), "'<!'": (72, 11), 'T.text_end': (72, 11), "'</'": (72, 11), "'<'": (72, 11)}, {}, ['element']), ({'T.text': (73, 11), '__end_of_input__': (73, 11), 'T.whitespace': (73, 11), "'<!'": (73, 11), 'T.text_end': (73, 11), "'</'": (73, 11), "'<'": (73, 11)}, {}, ['element']), ({'T.text': (73, 11), '__end_of_input__': (73, 11), 'T.whitespace': (73, 11), "'<!'": (73, 11), 'T.text_end': (73, 11), "'</'": (73, 11), "'<'": (73, 11)}, {}, ['element']), ({'T.text': (73, 11), '__end_of_input__': (73, 11), 'T.whitespace': (73, 11), "'<!'": (73, 11), 'T.text_end': (73, 11), "'</'": (73, 11), "'<'": (73, 11)}, {}, ['element']), ({'T.text': (73, 11), '__end_of_input__': (73, 11), 'T.whitespace': (73, 11), "'<!'": (73, 11), 'T.text_end': (73, 11), "'</'": (73, 11), "'<'": (73, 11)}, {}, ['element']), ({'T.text': (74, 11), '__end_of_input__': (74, 11), 'T.whitespace': (74, 11), "'<!'": (74, 11), 'T.text_end': (74, 11), "'</'": (74, 11), "'<'": (74, 11)}, {}, ['element']), ({'T.text': (74, 11), '__end_of_input__': (74, 11), 'T.whitespace': (74, 11), "'<!'": (74, 11), 'T.text_end': (74, 11), "'</'": (74, 11), "'<'": (74, 11)}, {}, ['element']), ({'T.text': (74, 11), '__end_of_input__': (74, 11), 'T.whitespace': (74, 11), "'<!'": (74, 11), 'T.text_end': (74, 11), "'</'": (74, 11), "'<'": (74, 11)}, {}, ['element']), ({'T.text': (74, 11), '__end_of_input__': (74, 11), 'T.whitespace': (74, 11), "'<!'": (74, 11), 'T.text_end': (74, 11), "'</'": (74, 11), "'<'": (74, 11)}, {}, ['element']), ({'T.text': (75, 11), '__end_of_input__': (75, 11), 'T.whitespace': (75, 11), "'<!'": (75, 11), 'T.text_end': (75, 11), "'</'": (75, 11), "'<'": (75, 11)}, {}, ['element']), ({'T.text': (75, 11), '__end_of_input__': (75, 11), 'T.whitespace': (75, 11), "'<!'": (75, 11), 'T.text_end': (75, 11), "'</'": (75, 11), "'<'": (75, 11)}, {}, ['element']), ({'T.text': (75, 11), '__end_of_input__': (75, 11), 'T.whitespace': (75, 11), "'<!'": (75, 11), 'T.text_end': (75, 11), "'</'": (75, 11), "'<'": (75, 11)}, {}, ['element']), ({'T.text': (75, 11), '__end_of_input__': (75, 11), 'T.whitespace': (75, 11), "'<!'": (75, 11), 'T.text_end': (75, 11), "'</'": (75, 11), "'<'": (75, 11)}, {}, ['element']), ({'T.text': (76, 11), '__end_of_input__': (76, 11), 'T.whitespace': (76, 11), "'<!'": (76, 11), 'T.text_end': (76, 11), "'</'": (76, 11), "'<'": (76, 11)}, {}, ['element']), ({'T.text': (76, 11), '__end_of_input__': (76, 11), 'T.whitespace': (76, 11), "'<!'": (76, 11), 'T.text_end': (76, 11), "'</'": (76, 11), "'<'": (76, 11)}, {}, ['element']), ({'T.text': (76, 11), '__end_of_input__': (76, 11), 'T.whitespace': (76, 11), "'<!'": (76, 11), 'T.text_end': (76, 11), "'</'": (76, 11), "'<'": (76, 11)}, {}, ['element']), ({'T.text': (76, 11), '__end_of_input__': (76, 11), 'T.whitespace': (76, 11), "'<!'": (76, 11), 'T.text_end': (76, 11), "'</'": (76, 11), "'<'": (76, 11)}, {}, ['element']), ({'T.text': (77, 11), '__end_of_input__': (77, 11), 'T.whitespace': (77, 11), "'<!'": (77, 11), 'T.text_end': (77, 11), "'</'": (77, 11), "'<'": (77, 11)}, {}, ['element']), ({'T.text': (77, 11), '__end_of_input__': (77, 11), 'T.whitespace': (77, 11), "'<!'": (77, 11), 'T.text_end': (77, 11), "'</'": (77, 11), "'<'": (77, 11)}, {}, ['element']), ({'T.text': (77, 11), '__end_of_input__': (77, 11), 'T.whitespace': (77, 11), "'<!'": (77, 11), 'T.text_end': (77, 11), "'</'": (77, 11), "'<'": (77, 11)}, {}, ['element']), ({'T.text': (77, 11), '__end_of_input__': (77, 11), 'T.whitespace': (77, 11), "'<!'": (77, 11), 'T.text_end': (77, 11), "'</'": (77, 11), "'<'": (77, 11)}, {}, ['element']), ({'T.text': (78, 11), '__end_of_input__': (78, 11), 'T.whitespace': (78, 11), "'<!'": (78, 11), 'T.text_end': (78, 11), "'</'": (78, 11), "'<'": (78, 11)}, {}, ['element']), ({'T.text': (78, 11), '__end_of_input__': (78, 11), 'T.whitespace': (78, 11), "'<!'": (78, 11), 'T.text_end': (78, 11), "'</'": (78, 11), "'<'": (78, 11)}, {}, ['element']), ({'T.text': (78, 11), '__end_of_input__': (78, 11), 'T.whitespace': (78, 11), "'<!'": (78, 11), 'T.text_end': (78, 11), "'</'": (78, 11), "'<'": (78, 11)}, {}, ['element']), ({'T.text': (78, 11), '__end_of_input__': (78, 11), 'T.whitespace': (78, 11), "'<!'": (78, 11), 'T.text_end': (78, 11), "'</'": (78, 11), "'<'": (78, 11)}, {}, ['element']), ({'T.text': (79, 11), '__end_of_input__': (79, 11), 'T.whitespace': (79, 11), "'<!'": (79, 11), 'T.text_end': (79, 11), "'</'": (79, 11), "'<'": (79, 11)}, {}, ['element']), ({'T.text': (79, 11), '__end_of_input__': (79, 11), 'T.whitespace': (79, 11), "'<!'": (79, 11), 'T.text_end': (79, 11), "'</'": (79, 11), "'<'": (79, 11)}, {}, ['element']), ({'T.text': (79, 11), '__end_of_input__': (79, 11), 'T.whitespace': (79, 11), "'<!'": (79, 11), 'T.text_end': (79, 11), "'</'": (79, 11), "'<'": (79, 11)}, {}, ['element']), ({'T.text': (79, 11), '__end_of_input__': (79, 11), 'T.whitespace': (79, 11), "'<!'": (79, 11), 'T.text_end': (79, 11), "'</'": (79, 11), "'<'": (79, 11)}, {}, ['element']), ({'T.text': (80, 11), '__end_of_input__': (80, 11), 'T.whitespace': (80, 11), "'<!'": (80, 11), 'T.text_end': (80, 11), "'</'": (80, 11), "'<'": (80, 11)}, {}, ['element']), ({'T.text': (80, 11), '__end_of_input__': (80, 11), 'T.whitespace': (80, 11), "'<!'": (80, 11), 'T.text_end': (80, 11), "'</'": (80, 11), "'<'": (80, 11)}, {}, ['element']), ({'T.text': (80, 11), '__end_of_input__': (80, 11), 'T.whitespace': (80, 11), "'<!'": (80, 11), 'T.text_end': (80, 11), "'</'": (80, 11), "'<'": (80, 11)}, {}, ['element']), ({'T.text': (80, 11), '__end_of_input__': (80, 11), 'T.whitespace': (80, 11), "'<!'": (80, 11), 'T.text_end': (80, 11), "'</'": (80, 11), "'<'": (80, 11)}, {}, ['element']), ({'T.text': (81, 11), '__end_of_input__': (81, 11), 'T.whitespace': (81, 11), "'<!'": (81, 11), 'T.text_end': (81, 11), "'</'": (81, 11), "'<'": (81, 11)}, {}, ['element']), ({'T.text': (81, 11), '__end_of_input__': (81, 11), 'T.whitespace': (81, 11), "'<!'": (81, 11), 'T.text_end': (81, 11), "'</'": (81, 11), "'<'": (81, 11)}, {}, ['element']), ({'T.text': (81, 11), '__end_of_input__': (81, 11), 'T.whitespace': (81, 11), "'<!'": (81, 11), 'T.text_end': (81, 11), "'</'": (81, 11), "'<'": (81, 11)}, {}, ['element']), ({'T.text': (81, 11), '__end_of_input__': (81, 11), 'T.whitespace': (81, 11), "'<!'": (81, 11), 'T.text_end': (81, 11), "'</'": (81, 11), "'<'": (81, 11)}, {}, ['element']), ({'T.text': (82, 11), '__end_of_input__': (82, 11), 'T.whitespace': (82, 11), "'<!'": (82, 11), 'T.text_end': (82, 11), "'</'": (82, 11), "'<'": (82, 11)}, {}, ['element']), ({'T.text': (82, 11), '__end_of_input__': (82, 11), 'T.whitespace': (82, 11), "'<!'": (82, 11), 'T.text_end': (82, 11), "'</'": (82, 11), "'<'": (82, 11)}, {}, ['element']), ({'T.text': (82, 11), '__end_of_input__': (82, 11), 'T.whitespace': (82, 11), "'<!'": (82, 11), 'T.text_end': (82, 11), "'</'": (82, 11), "'<'": (82, 11)}, {}, ['element']), ({'T.text': (82, 11), '__end_of_input__': (82, 11), 'T.whitespace': (82, 11), "'<!'": (82, 11), 'T.text_end': (82, 11), "'</'": (82, 11), "'<'": (82, 11)}, {}, ['element']), ({'T.text': (83, 11), '__end_of_input__': (83, 11), 'T.whitespace': (83, 11), "'<!'": (83, 11), 'T.text_end': (83, 11), "'</'": (83, 11), "'<'": (83, 11)}, {}, ['element']), ({'T.text': (83, 11), '__end_of_input__': (83, 11), 'T.whitespace': (83, 11), "'<!'": (83, 11), 'T.text_end': (83, 11), "'</'": (83, 11), "'<'": (83, 11)}, {}, ['element']), ({'T.text': (83, 11), '__end_of_input__': (83, 11), 'T.whitespace': (83, 11), "'<!'": (83, 11), 'T.text_end': (83, 11), "'</'": (83, 11), "'<'": (83, 11)}, {}, ['element']), ({'T.text': (83, 11), '__end_of_input__': (83, 11), 'T.whitespace': (83, 11), "'<!'": (83, 11), 'T.text_end': (83, 11), "'</'": (83, 11), "'<'": (83, 11)}, {}, ['element']), ({'T.text': (84, 11), '__end_of_input__': (84, 11), 'T.whitespace': (84, 11), "'<!'": (84, 11), 'T.text_end': (84, 11), "'</'": (84, 11), "'<'": (84, 11)}, {}, ['element']), ({'T.text': (84, 11), '__end_of_input__': (84, 11), 'T.whitespace': (84, 11), "'<!'": (84, 11), 'T.text_end': (84, 11), "'</'": (84, 11), "'<'": (84, 11)}, {}, ['element']), ({'T.text': (84, 11), '__end_of_input__': (84, 11), 'T.whitespace': (84, 11), "'<!'": (84, 11), 'T.text_end': (84, 11), "'</'": (84, 11), "'<'": (84, 11)}, {}, ['element']), ({'T.text': (84, 11), '__end_of_input__': (84, 11), 'T.whitespace': (84, 11), "'<!'": (84, 11), 'T.text_end': (84, 11), "'</'": (84, 11), "'<'": (84, 11)}, {}, ['element']), ({'T.text': (85, 11), '__end_of_input__': (85, 11), 'T.whitespace': (85, 11), "'<!'": (85, 11), 'T.text_end': (85, 11), "'</'": (85, 11), "'<'": (85, 11)}, {}, ['element']), ({'T.text': (85, 11), '__end_of_input__': (85, 11), 'T.whitespace': (85, 11), "'<!'": (85, 11), 'T.text_end': (85, 11), "'</'": (85, 11), "'<'": (85, 11)}, {}, ['element']), ({'T.text': (85, 11), '__end_of_input__': (85, 11), 'T.whitespace': (85, 11), "'<!'": (85, 11), 'T.text_end': (85, 11), "'</'": (85, 11), "'<'": (85, 11)}, {}, ['element']), ({'T.text': (85, 11), '__end_of_input__': (85, 11), 'T.whitespace': (85, 11), "'<!'": (85, 11), 'T.text_end': (85, 11), "'</'": (85, 11), "'<'": (85, 11)}, {}, ['element']), ({'T.text': (86, 11), '__end_of_input__': (86, 11), 'T.whitespace': (86, 11), "'<!'": (86, 11), 'T.text_end': (86, 11), "'</'": (86, 11), "'<'": (86, 11)}, {}, ['element']), ({'T.text': (86, 11), '__end_of_input__': (86, 11), 'T.whitespace': (86, 11), "'<!'": (86, 11), 'T.text_end': (86, 11), "'</'": (86, 11), "'<'": (86, 11)}, {}, ['element']), ({'T.text': (86, 11), '__end_of_input__': (86, 11), 'T.whitespace': (86, 11), "'<!'": (86, 11), 'T.text_end': (86, 11), "'</'": (86, 11), "'<'": (86, 11)}, {}, ['element']), ({'T.text': (86, 11), '__end_of_input__': (86, 11), 'T.whitespace': (86, 11), "'<!'": (86, 11), 'T.text_end': (86, 11), "'</'": (86, 11), "'<'": (86, 11)}, {}, ['element']), ({'T.whitespace': 423, 'T.tag_name': (32, 0, 'opt_space'), 'T.string': (32, 0, 'opt_space')}, {32: 1027}, ['attr']), ({'T.tag_name': 1028, 'T.string': 1029}, {}, ['attr']), ({'T.whitespace': (141, 5, 'attr'), "'/>'": (141, 5, 'attr'), "'>'": (141, 5, 'attr')}, {}, ['attr']), ({'T.whitespace': (141, 5, 'attr'), "'/>'": (141, 5, 'attr'), "'>'": (141, 5, 'attr')}, {}, ['attr']))

if __name__ == "__main__":
    main()

# vim: set shiftwidth=4 expandtab softtabstop=8 :
