# coding: utf-8
'''
Created on 12.04.2011

@author: prefer
'''
import ast
import os
import json
import codecs
import codegen
import shutil # Для копирования файлов

# Для тестов 
import pprint

from advanced_ast import StringSpaces


#===============================================================================
# Классы исключительных ситуаций
class ParserError(Exception):
    '''
    Возможные ошибки парсера
    '''
    def __init__(self, text=''):
        self.text = text

    def __str__(self):
        return self.text
    
    def __repr__(self):
        return self.text

class UndefinedGeneratedFunc(ParserError):
    '''
    Не определена функция автогенерации
    '''
    pass

#===============================================================================
class Parser(object):
    '''
    Класс, отвечающий за преобразование данных
    '''
    
    # Имя функция, которая будет сгенерирована
    GENERATED_FUNC = 'initialize'
    
    # Название папки для бакапа
    BACKUP_DIR_NAME = '.form-backup'
    
    # Сколько бакупных файлов хранить
    BACKUP_FILES_MAXIMUM = 10
    
    # Для юникода
    UNICODE_STR = '#coding: utf-8\n\n'
    
    # Док стринга для функции автогенерации
    GENERATED_FUNC_DOCSTRING = '''AUTOGENERATED'''
    
    # Откуда будут доступны все файлы
    IMPORT_ALL = 'm3.ui.ext.all_components'
    
    def __init__(self, path, class_name, func_name=None):
        '''
        @param path: Путь до py файла
        @param class_name: Имя класса для генерации
        '''
        
        # Путь до файла с исходным кодом
        self.path = path
        
        # Имя класса
        self.class_name = class_name

        # Имя функции
        self.func_name = func_name
        
        # Базовый класс для окон (Если нет в маппинге)
        self.base_class = 'ExtWindow'
                
        #
        self.base_class_name = None
        
    @staticmethod
    def get_source(path):
        '''
        Убирает символ "возврат каретки", оставляемый на win системах
        и убирает конечные пробелы
        '''        
        f_lines = map(lambda x: x.replace('\r', ''), open(path).xreadlines())
        lines = f_lines[:]

        f_lines.reverse()
        for line in f_lines:
            if str.isspace(line):
                lines.pop()
            else:
                break
                                    
        return ''.join(lines)                
    
    def to_designer(self, source_code=''):
        '''
        Отвечает за преобразования py-кода в json, понятный m3-дизайнеру.
        Возвращает json объект в виде строки.
        '''

        # AST не дружит в конечными пробельными строками, поэтому убираем их и все пробельные строки за одно
        source_code = Parser.get_source(self.path)

        try:
            node_module = ast.parse(source_code)
        except SyntaxError:
            raise ParserError('Некорректный синтаксис файла')

        class_node, func_node = Parser.get_func(node_module, self.class_name, self.func_name)
        
        if func_node.name == Parser.GENERATED_FUNC:
            self.base_class = self._get_base_class(class_node)
        else:
            if func_node.args.defaults:
                self.base_class = func_node.args.defaults[0].id
            else:
                raise ParserError('Синтаксис функции %s не поддерживается ' % str(self.func_name))
            
            # Название второго параметра, т.к. первый равен self
            self.base_class_name = func_node.args.args[1].id            
                   
        if not func_node:
            raise UndefinedGeneratedFunc('Функция автогенерации "%s" не определена в классе "%s"' % 
                                (Parser.GENERATED_FUNC, self.class_name)    )
              
        return self._parse_nodes(func_node)

    def _parse_nodes(self, func_node):
        '''
        Парсит файл
        '''
        
        self.config_cmp = {}
        self.extends_cmp = {}
        
        nodes = func_node.body
                
        for node in nodes:
            if isinstance(node, ast.Assign) and isinstance(node.value, ast.Name) and node.value.id not in ('True', 'False'):                                                               
                
                # Только в функции инициализации могут быть строки:
                # self.top_bar = my_top_bar
                # В других контейнерных функциях будет нечто ввиде:
                # cont.top_bar = my_top_bar
                if (not self.path or # Генерация js из пришедшего кода в "просмотрщике" \
                    isinstance(func_node, ast.FunctionDef) and  # Генерация js из файла \
                    func_node.name != Parser.GENERATED_FUNC) and \
                    'self' == node.targets[0].value.id:
                    continue
                
                # Составление структуры конфигов и типов компонентов
                parent, parent_item, item = self._get_attr(node)
                self.extends_cmp.setdefault(parent, {})[parent_item] =  item
            elif isinstance(node, ast.Assign):
                
                parent, attr, value = self._get_config_component(node)
                
                if value != self.base_class_name:
                    # Если это тип встроенной функции, то не нужно учитывать
                    self.config_cmp.setdefault(parent, {})[attr] = value 
                                                
            elif isinstance(node, ast.Expr) and isinstance(node.value, ast.Call) and node.value.args:

                parent, parent_item, items = self._get_extends(node.value)
                self.extends_cmp.setdefault(parent, {})[parent_item] =  items
            elif isinstance(node, ast.Expr) and isinstance(node.value, ast.Str):
                # док. строки 
                pass
            
            elif isinstance(node, ast.Return):
                # Функции не initialize
                pass
        
            else:
                raise ParserError("Синтаксис файла не поддерживается")
        
        # Находит корневой контейнер
        parent_key = self._get_parent_key(self.extends_cmp)
        
        
        if not parent_key:
            # Если вложенных контейнеров нет,  берем первую Assign конструкцию вида, например:
            # self.layout = 'auto', откуда parent_key = self
            
            for node in nodes:
                if isinstance(node, ast.Assign):
                    if isinstance(node.targets[0], ast.Name):
                        parent_key = node.targets[0].id
                        break
                    elif isinstance(node.targets[0], ast.Attribute):                    
                        parent_key = node.targets[0].value.id
                        break
            else:
                raise ParserError('Некорректный синтаксис файла')
        
        js_dict = {}         
        self._build_json(js_dict, parent_key)
            
        return js_dict                

    def _get_parent_key(self, extends_cmp):
        '''
        Возвращает корневой контейнер
        '''
        def check_key(verification_key):
            for _, v in extends_cmp.items():                
                if isinstance(v, dict):     
                    
                    # Вложенный словарь может содержать множество свойств.
                    # И пока здесь содержатся все атрибуты, который напрямую
                    # прописываются в self               
                    for item in v.values(): 
                        if verification_key in item:
                            return True
            else:
                return False
                     
        for k, _ in extends_cmp.items():            
            if not check_key(k):
                return k
     
         
    def _build_json(self, js_dict, key):
        '''
        Возвращает вложенный словарь объектов со своими свойствами
        '''
        item = self.extends_cmp.get(key)
        
        js_dict.update( self._get_json_config(key) )
        if isinstance(item, dict):
            for k, v in item.items():
                if isinstance(v, list):
                    
                    l = []
                    for value in v:                    

                        if self.extends_cmp.get(value):                            
                            p = {}
                            l.append(p)                   
                            self._build_json(p, value)
                        else:                            
                            l.append(self._get_json_config(value))                            
            
                    extjs_item = self._get_json_attr(k, js_dict['type'])                    
                    if not extjs_item:
                        raise ParserError('Не определен объект маппинга "%s" для класса "%s"' % ( str(k), js_dict['type']))
                                
                    js_dict[extjs_item] = l
                else: 
                    # Приходят property                    
                    extjs_item = self._get_json_attr(k, js_dict['type'])
                    if not extjs_item:                                                                    
                        # Если в маппинге нет такого объекта, то продолжаем дальше, т.к. нужно исключать объекты,
                        # которые напрямую прописываются в self 
                        
                        continue  

                    # Объект может быть вложенный
                    if self.extends_cmp.get(v):                            
                        p = {}                 
                        self._build_json(p, v)                                                
                        js_dict[extjs_item] = p
                    else:                                                    
                        js_dict[extjs_item] = self._get_json_config(v)                    
        
        
    def _get_extends(self, node):
        '''
        Возвращение вложенных объектов
        parent - Название объекта-контейнера
        parent_item - название поля, где находятся дочернии объекты
        items - список дочерних объектов 
        '''
        parent =  node.func.value.value.id
        parent_item =  node.func.value.attr
                
        if not isinstance(node.args[0], ast.List):
            raise ParserError('Cинтаксис класса "%s" не поддерживается' % self.class_name)
        
        items = list(map(lambda x: x.id, node.args[0].elts))                
        return parent, parent_item, items
    
    def _get_attr(self, node):
        '''
        Получение значения атрибуты
        parent: Родительский контейнер
        parent_item: Поле родительского контейнера
        value: Значение поля  
        '''
        value = node.value.id
        parent_item =  node.targets[0].attr
        parent =  node.targets[0].value.id
        return parent, parent_item, value

    def _get_base_class(self, class_node):
        '''
        Получает базовый класс
        '''
        base = class_node.bases[0]
        if isinstance(base, ast.Name):
            return class_node.bases[0].id
        elif isinstance(base, ast.Attribute):
            return class_node.bases[0].attr
        else:
            raise ParserError('Базовый класс компонента не поддерживается')
    
    def _get_json_config(self, key):
        '''
        Возвращает конфигурацию компонента в качестве словаря
        '''        
        properties, py_name = self._get_properties(key)

        properties['type'] = py_name
        properties['id'] = key
        return properties

    def _get_properties(self, key):
        '''
        Возвращает кортеж: свойства контрола, имя контрола в натации дизайнера
        (extjs)
        '''

        conf = self.config_cmp[key].copy()        
        py_name = conf.pop('py_name') if conf.get('py_name') else self.base_class
        
        extjs_name = self._get_extjs_class(py_name)

        if not extjs_name:
            raise ParserError('Не определен класс маппинга "%s"' % py_name)
        
        properties = dict(id=key) 
        for k, v in conf.items():            
            if isinstance(v, dict) and v.has_key('type'):                
                # Обрабатываются комплексные компоненты
                extjs_attr, value = self._get_json_complex_attr(k, v, extjs_name)
                
                properties[extjs_attr] = value
            else:              
                extjs_attr = self._get_json_attr(k, extjs_name)            
                if not extjs_attr:                    
                    raise ParserError('Не определен объект маппинга "%s" для класса "%s"' % ( str(k), extjs_name))
                  
                properties[extjs_attr] = v
            
        return properties, extjs_name
    
    def _get_json_complex_attr(self, name, value, extjs_class_name):
        '''
        Получается значение для комплексного компонента
        '''
        conf = self._gen_config(extjs_class_name)       
 
        for k, v in conf.items():                
            if isinstance(v, dict) and name == v['value'] and v['type'] == value['type']:                
                return str(k), value['value']
        else:
            raise ParserError('Комплексное свойство %s в классе %s не найдено' % (name, extjs_class_name) )
        
    def _get_extjs_class(self, py_name):
        '''
        Получает из маппинга наименование extjs класса 
        '''
        for item in self._get_mapping():
            k, v = item['class'].items()[0] # Одно значение
            if v == py_name:
                return str(k)
                    
    def _get_json_attr(self, name, extjs_class_name):
        '''
        Получает из маппинга свойство по наименованию extjs контрола
        '''
        conf = self._gen_config(extjs_class_name)        
        for k, v in conf.items():
            if v == name:
                return str(k)
        
    def _gen_config(self, type_obj):  
        '''
        Получение конфигурации свойств из маппинга по типу объекта
           @param type_obj: Тип объекта (window, panel, etc.)
        '''         
        for item in self._get_mapping():
            if item['class'].has_key(type_obj):
                return item['config']        

    def _get_value(self, node):
        '''
        Получает значение исходя из типа узла
        '''
        return ast.literal_eval(node)

    def get_complex_property(self, node):
        '''
        Пробует распарсить комплексное свойство
        '''        
        if isinstance(node, ast.Attribute) and isinstance(node.value, ast.Call) \
            and len(node.value.args) == 1 and isinstance(node.value.args[0], ast.Str):     
            # Маппинг для shortname
             
            # Чтобы потом при подстановки extjs названия полей выяснить, что
            # это поле нужно маппить отдельно                  
            return dict(type='shortname',value=node.value.args[0].s)
        else:
            raise ParserError('Некорректный синтаксис файла')


    def _get_config_component(self, node):
        '''
        Разбирает конструкцию вида:
        
        self.width = 100
        
        parent - self
        attr - width
        value - 100        
        '''        
        if not isinstance(node, ast.Assign):
            raise ParserError("Некорректный синтаксис файла")
        
        if isinstance(node.value, ast.Call):
            # Создание экземпляра            
            # instanse, attr, class name
            
            func = node.value.func
            if isinstance(func, ast.Attribute):
                if isinstance(node.targets[0], ast.Name):
                    return node.targets[0].id, 'py_name', '%s.%s' % (func.value.id ,func.attr)
                else:
                    # Пробуем разобрать комплексное свойство
                    
                    parent, attr =  node.targets[0].value.id, node.targets[0].attr                    
                    value = self.get_complex_property(func)
                    return parent, attr, value
                    
            elif isinstance(node.value.func, ast.Name):                
                return node.targets[0].id, 'py_name', func.id
            else:
                raise ParserError("Некорректный синтаксис файла")
        else:            
            # Распарсивание свойства
            # parent, attr, value
            return node.targets[0].value.id, node.targets[0].attr, self._get_value(node.value)        
        
    def _create_insanse_node(self, id, instanse_name):
        return ast.Assign([ast.Name( 
                        id, 
                        ast.Load()
                  )], 
                  ast.Call(
                        ast.Name( 
                            instanse_name, 
                            ast.Load()
                        ), 
                        [], 
                        [], 
                        None, 
                        None)
                  )
    
    def from_designer(self, json_dict):
        '''
        Отвечает за преобразование json-a формы из m3-дизайнеру в py-код.
        '''

        # AST не дружит в конечными пробельными строками, поэтому убираем их и все пробельные строки за одно
        old_source_code = Parser.get_source(self.path)
        
        try:
            module_node = ast.parse(old_source_code)
        except SyntaxError:
            raise ParserError('Некорректный синтаксис файла')
        # Нахождение нужной функции GENERATED_FUNC        
        class_node, func_node = Parser.get_func(module_node, self.class_name, self.func_name)
        nodes = Node(json_dict).get_nodes(self._get_mapping())
        # self._set_class_name(class_node, json_dict['type']) -- Не используется в данный момент
        
        # Старая док строка не должна потеряться
        if func_node and isinstance(func_node.body, list) and len(func_node.body) > 0 \
            and isinstance(func_node.body[0], ast.Expr):            
            nodes.insert(0, func_node.body[0])
        
        if self.func_name and self.func_name != Parser.GENERATED_FUNC:
            nodes.insert(0, self._create_insanse_node(json_dict['id'], func_node.args.args[1].id))
            nodes.extend([StringSpaces(), ast.Return(value=ast.Name(id=json_dict['id'], ctx=ast.Load()))])
            
        # Замена старого содержимого на новое сгенерированное 
        func_node.body = nodes + [StringSpaces(lines=2), ]
                        
        begin_line, end_line = Parser._get_number_lines(module_node, class_node, func_node)

        # Бакап файла на всякий пожарный случай и cохранение нового файла
        source_code = codegen.to_source(func_node, indentation=1)
        
        self._write_to_file(source_code, begin_line, end_line)
        
    @staticmethod
    def _get_number_lines(module_node, class_node, func_node):
        '''
        Возвращает начало и конец функции Parser.GENERATED_FUNC_DOCSTRING
        '''
        begin_line = func_node.lineno
        
        # А вот с концом функции немного сложнее
        # Сначало проверим, есть ли еще методы в данном классе
        # Если нет, то то возьмем следующий узел module_node после класса
        for i, node in enumerate(class_node.body):            
            if node == func_node and len(class_node.body) > i+1:
                return begin_line, class_node.body[i+1].lineno
        else:
            for j, node in enumerate(module_node.body):
                if node == class_node and len(module_node.body) > j+1:
                    return begin_line, module_node.body[j+1].lineno
            else:
                return begin_line, None
        
    def to_designer_preview(self, source_code):
        '''
        Возвращает по py-коду js-представление
        '''
        try:
            nodes = ast.parse(source_code)
        except SyntaxError:
            raise ParserError('Некорректный синтаксис файла')
                
        return self._parse_nodes(nodes)

    
    @staticmethod
    def from_designer_preview(json_dict):
        '''
        Преобразовывает js в py код и возвращает его в виде строки
        '''
        
        # Напрямую сбиндим объект маппинга
        nodes = Node(json_dict).get_nodes( mapping_list )

        list_source_code = map(codegen.to_source, nodes)
        return '\n'.join(list_source_code)

 
    def _set_class_name(self, node, extjs_type):
        '''
        Устанавливает соответсвующее имя класса наследника
        '''
        for item in self._get_mapping():
            k, v = item['class'].items()[0]
            if k ==  extjs_type:
                node.bases = [ast.Name( str(v), ast.Load())] 
                break
        
    def _write_to_file(self, source_code, begin_line, end_line):
        '''
        Запись в файл сгенерированного кода
        '''
        # 1. Создание директории, если нет
        # 2. Рекурсивное переименование всех имеющихся бакупных файлов в
        # файлы вида: .old.1, old.2, old.3
        # 3. Сохранение старого исходного кода с расширением old
        # 4. Перезапись файла новым содержимым

        dir = os.path.dirname(self.path)
        dir_backup = os.path.join(dir, Parser.BACKUP_DIR_NAME)
        if not os.path.isdir(dir_backup):
            os.mkdir(dir_backup)
        else:
            # Могут быть бакупные файлы, обход по времени последнего изменения
            for file_name in sorted(os.listdir(dir_backup),
                                key=lambda f: os.stat(  os.path.join(dir_backup, f) ).st_mtime):
                file_path = os.path.join(dir_backup, file_name)

                i = file_path.split('.')[-1]
                try:
                    int(i)
                except ValueError:
                    os.rename(file_path, '%s.%d' % (file_path, 0) )
                else:
                    if int(i)+1 > Parser.BACKUP_FILES_MAXIMUM:
                        os.remove(file_path)
                    else:
                        file_name_parts = file_path.split('.')
                        without_end = file_name_parts[:-1]
                        os.rename(file_path, '%s.%d' % ('.'.join(without_end), int(i)+1) )

        new_path = os.path.join(dir_backup, os.path.basename(self.path))
        shutil.copyfile(self.path, new_path + '.old')

        f_lines = open(self.path, 'r').readlines()
        
        # Включаем и свою строку и пробельную строку выше
        end_l = f_lines[end_line-2:] if end_line else []

        f_line = open(self.path, 'rb').readline()

        #файл пишется как бинарный "wb" и при сохранении не добавляется лишний знак перенова (CR)
        with open(self.path, 'w' if '\r' in f_line else 'wb') as f:
            f.write(''.join(f_lines[:begin_line-1]) + source_code + ''.join(end_l))

    def _get_mapping(self):
        '''
        Получение объекта маппинга
        '''
        return mapping_list
        
    @staticmethod
    def get_func(node_module, class_name, func_name):
        '''
        Поиск и возвращение функции GENERATED_FUNC 
        '''

        func_name = func_name or Parser.GENERATED_FUNC

        for node in node_module.body:        
            if isinstance(node, ast.ClassDef) and node.name == class_name:                
                for nested_node in node.body:    
                    if isinstance(nested_node, ast.FunctionDef) and \
                            nested_node.name == func_name:
                        return node, nested_node
                else:
                    raise UndefinedGeneratedFunc('Функция автогенерации "%s" не \
                        определена в классе "%s"' % (Parser.GENERATED_FUNC, str(class_name)))
                    
    @staticmethod
    def generate_class(class_name, class_base):
        '''
        Генерирует класс
        Конструкция вида:
        class <class_name>(<class_base>):
            def __init__(self, *args, **kwargs):
                super(<class_name>).__init__(*args, **kwargs)
                self.<Parser.GENERATED_FUNC>()
                
            def <Parser.GENERATED_FUNC>(self):
                self.layout = 'auto'
        
        Возвращает сгенерированный узел AST
        '''
        # Лисп ёпта
        node_constructor = ast.Expr(
                                ast.Call(
                                    ast.Attribute(
                                        ast.Call(
                                            ast.Name('super', 
                                                     ast.Load()
                                            ),
                                            [ast.Name( 
                                                str(class_name),
                                                ast.Load()
                                             ),
                                             ast.Name(
                                                 'self',
                                                 ast.Load()
                                             )],
                                            [],
                                            None,
                                            None
                                        )
                                         ,'__init__',
                                         ast.Load()
                                    )
                                    ,[]
                                    ,[]
                                    ,ast.Name(
                                        'args',
                                        ast.Load()
                                    ),
                                    ast.Name(
                                        'kwargs',
                                        ast.Load()
                                    )
                                )
                            )
        
        node_initial = ast.Expr(
                                ast.Call(
                                    ast.Attribute(
                                        ast.Name(
                                            'self',
                                            ast.Load()
                                        ),
                                        Parser.GENERATED_FUNC,
                                        ast.Load()                                              
                                    ),
                                    [],     
                                    [],
                                    None,
                                    None,
                                )
                            )
    
        contstructor_args = ast.arguments([ast.Name('self', ast.Load())], 
                                          'args', 
                                          'kwargs', 
                                          [])    
        
        constructor_func = ast.FunctionDef('__init__',
                                            contstructor_args, 
                                            [node_constructor, node_initial], 
                                            [])
    
        initialize_func = Parser.generate_initialize()
            
        cl = ast.ClassDef(
                    str(class_name), 
                    [ast.Name(class_base, ast.Load())], 
                    [constructor_func, initialize_func,], 
                    [])
        
        return cl
    
    @staticmethod
    def generate_initialize():
        '''
        Генерирует AST узел для функции автогенерации
        '''        
        initial_args = ast.arguments([ast.Name('self', ast.Load())], 
                             None, 
                             None, 
                             [])
            
        initial_nodes = [ast.Assign(
                                [ast.Attribute(
                                    ast.Name('self', ast.Load()), 
                                    'layout', 
                                    ast.Load()
                                  )],
                                 ast.Str('auto')) ]
        
        doc_str = ast.Expr(
            ast.Str(Parser.GENERATED_FUNC_DOCSTRING)
            )
        
        initialize_func = ast.FunctionDef(Parser.GENERATED_FUNC, 
                          initial_args, 
                          [doc_str,] + initial_nodes, 
                          [])
        
        return initialize_func
    
    @staticmethod
    def generate_cont_func(name_func, type_func_ext, name_instanse='cont'):
        '''
        Генерирует контейнерную функцию вида:
        
        def <name_func>(self, clazz=<type_func_py>):
            cont = clazz()
            cont.layout = 'auto'
            
            return cont
        '''
        #TODO: вынести в общую функцию
        def get_py_name(extjs_name):
            for item in mapping_list:                
                if item['class'].has_key(extjs_name):
                    return item['class'][extjs_name]
            else:
                raise ParserError(u'Не определен объект маппинга %s' % extjs_name)
        
        param_name = 'container_class'
        
        args = ast.arguments(args=[
                                   ast.Name('self', ast.Load()), 
                                   ast.Name(param_name, ast.Load())
                             ],
                             defaults=[ast.Name( str(get_py_name(type_func_ext)) , ast.Load())], 
                             kwarg=None, 
                             vararg=None)
            
        nodes = [ast.Assign(targets=[ast.Name('cont', ast.Load())], 
                            value=ast.Call(
                                    ast.Name(
                                        param_name,
                                        ast.Load()                                      
                                    ),
                                    [],     
                                    [],
                                    None,
                                    None,
                                )),
                 
                 
                 ast.Assign(
                    [ast.Attribute(
                        ast.Name(name_instanse, ast.Load()), 
                        'layout', 
                        ast.Load()
                      )],
                     ast.Str('auto')),
                 
                 StringSpaces(),
                 ast.Return(ast.Name(name_instanse, ast.Load()))
                 ]
        
        doc_str = ast.Expr(
            ast.Str(Parser.GENERATED_FUNC_DOCSTRING)
            )
        
        initialize_func = ast.FunctionDef(str(name_func), 
                              args, 
                              [doc_str,] + nodes, 
                              []
                          )
        
        return initialize_func
    
    @staticmethod
    def generate_import():
        '''
        Генерирует узел AST, который преобразуется в 
        
        from m3.ui.ext.all_components import *
        '''
        return ast.ImportFrom(Parser.IMPORT_ALL, [ast.alias(name='*')], 0)
        
class Node(object):
    '''
    Обрабатывает данные, пришедшие из дизайнера и возвращает список ast узлов
    '''
    
    # Ссылка на маппинг
    mapping = None
 
    def __init__(self, data):
        '''
        @param data: Словарь, которые нужно представить как ast узлы
        '''   
        self.data = data                            

    @staticmethod
    def sort_items(items):
        '''
        Сортировка по приоритеты: 
        1. Список
        2. Словарь
        3. Если ключ является id
        4. Все остальное 
        '''
        key, item = items
        if isinstance(item, list):
            return 100
        elif isinstance(item, dict):
            return 10
        elif key == 'id':            
            return 0
        else:            
            return 1


    def get_nodes(self, mapping):
        '''
        Генерирует и возвращает ast узлы
        '''
        Node.mapping = mapping
        
        nodes = []
        nodes_attr = []
        nodes_extends = [] 
        nodes_in_self = []
        
        self.walk(nodes, nodes_attr, nodes_extends, nodes_in_self)
        
        if nodes_attr:
            nodes_attr.insert(0, StringSpaces())
            nodes += nodes_attr
            
        if nodes_extends:
            nodes_extends.insert(0, StringSpaces())
            nodes += nodes_extends
            
        if nodes_in_self:
            nodes_in_self.insert(0, StringSpaces())
            nodes += nodes_in_self
        
        return nodes

    def walk(self, nodes, nodes_attr, nodes_extends, nodes_in_self, deep=0):        
        '''
        Рекурсивно спускается по вложенному словарю и заполняет структуры данными
        @param nodes: Общий список ast узлов
        @param nodes_attr: Список ast узлов для компонентов один-к-одному (grid.store = my_store)
        @param nodes_extends: Список ast узлов, для компонентов один-ко-многим 
            (grid.columns.extend([grid_column1, grid_column2, grid_column3,]))
        @param nodes_in_self: Список ast узлов, которые должны прописаться в self
        @param deep: Глубина рекурсии для вычисления родительского компонента
        '''
        for key, value in sorted(self.data.items(), key=Node.sort_items):
            if isinstance(value, list):
                extends_list = []
                for item in value:
                    if isinstance(item, dict) and item.has_key('id') and item.has_key('type'):
                        Node(item).walk(nodes, nodes_attr, nodes_extends, nodes_in_self, deep+1)
                        extends_list.append(item['id'])
                    else:
                        ast_node = self._get_property(self.data['id'], key, value, self.data['type'])
                        nodes.append(ast_node)
                        break
                
                if extends_list:
                    ast_node = self._get_extends(self.data['id'], key, extends_list, self.data['type'])                    
                    nodes_extends.append(ast_node)
            elif isinstance(value, dict) and value.has_key('type') and value.has_key('id'):                
                Node(value).walk(nodes, nodes_attr, nodes_extends, nodes_in_self, deep+1)             
                
                ast_node = self._get_attr(self.data['id'], key, value['id'], self.data['type'])
                
                nodes_attr.append(ast_node)
            else:                
                if key == 'type' or (deep == 0 and key == 'id'):                    
                    continue
                elif key == 'id' and deep > 0:                                                   
                    ast_node = self._get_instanse(self.data['type'], value)                                        
                    nodes.extend([StringSpaces(), ast_node, ])
                    
                    in_self_node = self._add_cmp_in_self(value)                    
                    nodes_in_self.append(in_self_node)                                
                else:                    
                    ast_node = self._get_property(self.data['id'], key, value, self.data['type'])
                    nodes.append(ast_node)                            
     
    def _add_cmp_in_self(self, field, parent_fields='self'):
        '''
        Генерирует узлы, который добавляют компонента внутрь self
        
        Например:
        self.my_simple_form = my_simple_form
        где:
        @param field:  my_simple_form        
        '''
        return ast.Assign(
                [ast.Attribute(
                        ast.Name(parent_fields, ast.Load()), 
                        str(field), 
                        ast.Load()
                    )], 
                ast.Name(field, ast.Load())
            )
    
    def _get_complex_field(self, py_attr, value):
        '''
        Преобразует сложные проперти
        Пока шортнеймы
        '''
        if py_attr['type'] == 'shortname':
            # Преобразует шортнэймы
            attr = py_attr['value']
            
            func = ast.Call(
                        func=ast.Attribute(
                            attr='absolute_url'
                            ,value=ast.Call(
                                    args=[ast.Str(value)]
                                    ,func=ast.Attribute(
                                        attr='get_action'
                                        ,value=ast.Name(
                                            id='urls'
                                            ,ctx=ast.Load()      
                                        )
                                        ,ctx=ast.Load()
                                    )                                    
                                    ,keywords=[]
                                    ,kwargs=None
                                    ,starargs=None  
                                )
                            ,ctx=ast.Load()                  
                        )
                        ,args=[]
                        ,keywords=[]
                        ,kwargs=None
                        ,starargs=None                       
                    ) 
            
            
            return str(attr), func
        else:
            raise ParserError('Комплексное свойство "%s" не поддерживается') % py_attr['type']
    
    def _get_property(self, parent_field, extjs_attr, value, extjs_class):
        '''
        Генерация связи один ко одному с другим компонентом, например:
        grid.store = my_data_store
        где:
        @param parent_field: grid
        @param extjs_attr: store в натации extjs
        @param value: my_data_store
        @param extjs_class: Название класса, в котором нужно провдить поиск 
        '''
        for item in self.mapping:
            if item['class'].has_key(extjs_class):
                                
                if not item['config'].get(extjs_attr):
                    raise ParserError('Не определен объект маппинга "%s" для класса "%s"' % ( str(extjs_attr) , extjs_class))
                
                py_attr = item['config'][extjs_attr]
                
                if isinstance(py_attr, dict):
                    # Сложное свойство
                    
                    attr, attr_value = self._get_complex_field(py_attr, value)
                    
                    return ast.Assign(
                        [ast.Attribute(
                            ast.Name(parent_field, ast.Load()), 
                            attr, 
                            ast.Load()
                        )], 
                        attr_value
                    )
                else:
                    return ast.Assign(
                        [ast.Attribute(
                            ast.Name(parent_field, ast.Load()), 
                            str(py_attr), 
                            ast.Load()
                        )], 
                        self._get_node_value(value)
                    )
        else:
            raise ParserError("Не определен объект маппинга для класса '%s'" % extjs_class) 
    
    def _get_instanse(self, extjs_class, value):
        '''
        Генерирует конструкцию вида:
        panel = ExtPanel()
        @param extjs_class: Класс в натации extjs, который преобразуется в ExtPanel
        @param value: panel 
        '''
        for item in self.mapping:
            if item['class'].has_key(extjs_class):
                instanse_name = item['class'][ extjs_class ]
                                                           
                if value.find(" ") > 0:
                    raise ParserError('Переменная "%s" не может содержать пробелы' % value)
                
                return ast.Assign([ast.Name( 
                                        value, 
                                        ast.Load()
                                  )], 
                                  ast.Call(
                                        ast.Name( 
                                            str(instanse_name) , 
                                            ast.Load()
                                        ), 
                                        [], 
                                        [], 
                                        None, 
                                        None)
                                  )
        else:
            raise ParserError("Не определен объект маппинга для класса '%s'" % extjs_class) 
    
    def _get_attr(self, parent_field, extjs_attr, value, extjs_class):
        '''
        Генерирует узел для свойства объекта
        Например: panel.layout = "auto"
        где:
        @param parent_field: panel в натации extjs
        @param extjs_attr: layout в натации extjs
        @param value: "auto"
        @param extjs_class: Название класса, в котором нужно провдить поиск 
        '''
        for item in self.mapping:
            if item['class'].has_key(extjs_class):
                                
                if not item['config'].get(extjs_attr):
                    raise ParserError('Не определен объект маппинга "%s" для класса "%s"' % ( str(extjs_attr), extjs_class))
                
                py_attr = item['config'][extjs_attr]
                return ast.Assign(
                    [ast.Attribute(
                        ast.Name(parent_field, ast.Load()), 
                        str(py_attr), 
                        ast.Load()
                    )], 
                    ast.Name(value, ast.Load())
                )
        else:            
            raise ParserError('Не определен класс маппинга "%s"' % extjs_class)
    

    def _get_extends(self, parent_field, extjs_name, list_cmp, extjs_class):      
        '''
        Генерирует узел для вложенных объектов
        Например: panel.items.extend([item1, item2, item3])
        где:
        @param parent_field: panel в натации extjs
        @param extjs_name: items в натации extjs 
        @param extjs_class: Класс, в котором нужно проводить поиск
        @param list_cmp: [item1, item2, item3] Список вложенных компонентов
        '''  
        for item in self.mapping:
            if item['class'].has_key(extjs_class):
                                
                if not item['config'].get(extjs_name):
                    raise ParserError('Не определен объект маппинга "%s" для класса "%s"' % ( str(extjs_name), extjs_class))                
                
                py_attr = item['config'][extjs_name]
                 
                ast_list = ast.List([ast.Name(item, ast.Load() ) for item in list_cmp], ast.Load())                                                                
                return ast.Expr( 
                            ast.Call(
                                ast.Attribute(
                                    ast.Attribute(
                                        ast.Name(
                                            parent_field,
                                            ast.Load()
                                        ),
                                        str(py_attr),
                                        ast.Load()       
                                    ),
                                    'extend',
                                    ast.Load()          
                                ),     
                                [ast_list,],
                                [],     
                                None,
                                None
                            )
                        )
        else:
            raise ValueError("Mapping is undefined for class '%s'" % extjs_class)            

    def _get_node_value(self, value):
        '''
        Генерация узла дерева для простых элементов
        Например для строки и числа, булевого типа
        '''        
        if value in ('False', 'True'):
            return ast.Name(value, ast.Load())
        elif isinstance(value, int):
            return ast.Num(value)
        elif isinstance(value, basestring):
            return ast.Str(value)
        elif isinstance(value, dict):
            return ast.Dict( [ self._get_node_value(str(k))  for k in value.keys()],
                             [ self._get_node_value(v) for v in value.values()] )
            
        elif isinstance(value, tuple):
            return ast.Tuple([self._get_node_value(item) for item in value], ast.Load())
               
        elif isinstance(value, list):
            return ast.List([self._get_node_value(item) for item in value], ast.Load())
        
        raise ParserError('Некорректный синтаксис: тип "%s" со значением "%s" не поддерживается' % (type(value), value))                     
                    
def update_with_inheritance(m_list, parent=None, config=None):
    '''
    Обновляет маппинг объектов с учетом наследования, то есть дозополняет
    свойствами объекты из унаследованных классов "parent"
    '''        
    
    for item in m_list:
        if parent and item.get('class').keys()[0] == parent:    
            
            # Для того, чтобы родительские свойства не перекрывали свойства объекта
            tmp_dict = config.copy() 
            config.update(item['config'])
            config.update(tmp_dict)
            break
        elif not parent and item.get('parent'):
            update_with_inheritance(m_list, item.get('parent'), item['config'])


def open_text_file(filename, mode='r', encoding = 'utf-8'):
    '''
    Для открытия файла, если он был сохранен под виндами. Бомы всякие удаляются.
    '''
    has_BOM = False
    if os.path.isfile(filename):
        f = open(filename,'rb')
        header = f.read(4)
        f.close()
        
        # Don't change this to a map, because it is ordered
        encodings = [ ( codecs.BOM_UTF32, 'utf-32' ),
            ( codecs.BOM_UTF16, 'utf-16' ),
            ( codecs.BOM_UTF8, 'utf-8' ) ]
        
        for h, e in encodings:
            if header.startswith(h):
                encoding = e
                has_BOM = True
                break
        
    f = codecs.open(filename,mode,encoding)
    # Eat the byte order mark
    if has_BOM:
        f.read(1)
        
    return f

def get_mapping():
    '''
    Возвращает строковое представление маппинга
    '''
    f = open_text_file(os.path.join(os.path.dirname(__file__), 'mapping.json'))
    return ''.join(filter(lambda x: not '//' in x, f.xreadlines()))

# Словарь сопоставлений контролов в дизайнере к контролам в питоне
mapping_list = json.loads( get_mapping() )

# Рекурсивное добавление свойств у классов наследников
update_with_inheritance(mapping_list)

#===============================================================================
def test_from_designer():
    '''
    С новым протоколом
    '''
    fake_data = {
        'name':'Ext window',
        'title':'Trololo',
        'layout':'fit',

        'type':'window',
        'id':'self',
        
        'items': [{
                'type': 'panel',
                'id': 'base_panel',
                
                'name':'Ext panel',
                'title':'Im panel ',
                'layout':'absolute',
                'items': [{
                    'type': 'gridPanel',
                    'id': 'grid_1',                      
                    'store': {
                        'id': 'store1'
                        ,'type':'arrayStore'
                    },
                      
                    'tbar':{
                            'id': 'tbar_1'
                            ,'type':'toolbar'
                            ,'items':[{
                                'type':'button'
                                ,'id':'button_1'
                                ,'text':u'Кнопка 1'
                            },{
                               'type':'button'
                                ,'id':'button_2'
                                ,'text':u'Кнопка 2'                            
                            }]
                    }
                    ,'columns': [{
                        'type': 'gridColumn',
                        'id': 'gridColumn_1',
                        'header': '2'  
                    },{
                       'type': 'gridColumn',
                        'id': 'gridColumn_2',
                        'header': '1'
                    }]     
                           
                }]
            }]
    }
        
    Node.mapping = mapping_list
    nodes = Node(fake_data).walk()

    list_nodes_str = map(codegen.to_source, nodes)
    print '\n'.join(list_nodes_str)         
    print '====== from_designer - ok ======'    
    
def test_to_designer(class_name='TestOne'):
    '''
    Для тестирования метода to_designer
    '''
    js = Parser('tests.py', class_name).to_designer()        
    pprint.pprint( js ) 
    
    print 'Parser.to_designer - ok'
