'''
    mania.ast
    ~~~~~~~~~

    :copyright: 2010 by Bjoern Schulz <bjoern.schulz.92@gmail.com>
    :license: MIT, see LICENSE for more details
'''

from itertools import izip
import os.path
from pkg_resources import resource_string
from mania.env import Environment
import mania.types

class Node(object):
    pass

class Sequence(Node):
    
    def __init__(self, *body):
        self.body = body
    
    def eval(self, env):
        result = mania.types.nil
        for node in self.body:
            result = node.eval(env)
        return result

class Value(Node):
    
    def __init__(self, value):
        self.value = value
    
    def eval(self, env):
        return self.value

class Reference(Node):
    
    def __init__(self, name):
        self.name = name
    
    def eval(self, env):
        return env.lookup(self.name)

class Assign(Node):
    
    def __init__(self, name, value):
        self.name, self.value = name, value
    
    def eval(self, env):
        return env.define(self.name, self.value.eval(env))

class List(Node):
    
    def __init__(self, value):
        self.value = value
    
    def eval(self, env):
        lst = []
        for item in self.value:
            if isinstance(item, ArgumentList):
                lst.extend(item.eval(env))
            else:
                lst.append(item.eval(env))
        return mania.types.List(lst)

class Struct(Node):
    
    def __init__(self, value):
        self.value = value

    def eval(self, env):
        dct = {}
        for key, value in self.value:
            dct[key] = value.eval(env)
        return mania.types.Struct(dct)

class GetAttribute(Node):
    
    def __init__(self, object, name):
        self.object, self.name = object, name

    def eval(self, env):
        return self.object.eval(env).get_attribute(self.name)

class SetAttribute(Node):
    
    def __init__(self, object, name, value):
        self.object, self.name, self.value = object, name, value
    
    def eval(self, env):
        return self.object.eval(env).set_attribute(
            self.name, self.value.eval(env)
        )

class DelAttribute(Node):
    
    def __init__(self, object, name):
        self.object, self.name = object, name

    def eval(self, env):
        return self.object.eval(env).del_attribute(self.name)

class Lambda(Node):
    
    def __init__(self, args, body):
        self.args, self.body = args, body
    
    def eval(self, env):
        return mania.types.Function(self.args, self.body, env.clone())

class ArgumentList(Node):
    
    def __init__(self, value):
        self.value = value
    
    def eval(self, env):
        return (
            self.value.eval(env) if isinstance(self.value, Node)
            else self.value
        )

class OptionalArgument(Node):
    
    def __init__(self, name):
        self.name = name
    
    def eval(self, env):
        return self.name

class Let(Node):
    
    def __init__(self, args, body):
        self.args, self.body = args, body
    
    def eval(self, env):
        env = env.clone()
        for name, value in self.args:
            env.define(name, value.eval(env))
        return self.body.eval(env)

class NamedLet(Node):
    
    def __init__(self, name, args, body):
        self.name, self.args, self.body, self.env = name, args, body, None
    
    def call(self, *args):
        env = self.env.clone()
        for i, (name, value) in enumerate(self.args):
            if len(args) > i:
                env.define(name, args[i])
            else:
                env.define(name, value.eval(env))
        return self.body.eval(env)
    
    def eval(self, env):
        self.env = Environment(env, {self.name: self})
        return self.call(*[v.eval(self.env) for k, v in self.args])

class Call(Node):
    
    def __init__(self, func, args):
        self.func, self.args = func, args
    
    def eval(self, env):
        args = []
        for arg in self.args:
            if isinstance(arg, ArgumentList):
                args.extend([a for a in arg.eval(env)])
                break
            else:
                args.append(arg.eval(env))
        return self.func.eval(env).call(*args)

class If(Node):
    
    def __init__(self, cond, positive, negative=None):
        self.cond, self.positive, self.negative = cond, positive, negative
    
    def eval(self, env):
        if self.cond.eval(env):
            return self.positive.eval(env)
        elif self.negative is not None:
            return self.negative.eval(env)

class Cond(Node):
    
    def __init__(self, cases, default=None):
        self.cases, self.default = cases, default
    
    def eval(self, env):
        for condition, body in self.cases:
            if condition.eval(env):
                return body.eval(env)
        if self.default is not None:
            return self.default.eval(env)

class And(Node):
    
    def __init__(self, left, right):
        self.left, self.right = left, right
    
    def eval(self, env):
        left = self.left.eval(env)
        if left.to_boolean():
            return self.right.eval(env)
        return left

class Or(Node):
    
    def __init__(self, left, right):
        self.left, self.right = left, right
    
    def eval(self, env):
        left = self.left.eval(env)
        if left.to_boolean():
            return left
        return self.right.eval(env)

class Require(Node):
    
    def __init__(self, path):
        self.path = path
    
    def eval(self, env):
        if env.engine.sandbox:
            raise ImportError('Mania is runnung in sandbox mode')
        name = self.path.eval(env).to_native()
        path = os.path.join(*name.split('.')) + '.mania'
        try:
            if os.path.exists(os.path.abspath(path)):
                with open(os.path.abspath(path)) as f:
                    src = f.read()
            else:
                src = resource_string(mania.__name__, 'lib/%s' % path)
        except IOError:
            raise ImportError('No module named %s' % name)
        e = Environment(None, env.engine.builtins).clone()
        mania.compile(src).eval(e)
        mod = {}
        for key, value in e.locals.items():
            mod[key] = value
        return mania.types.Module(name, mod)