'''
    mania.builtins
    ~~~~~~~~~~~~~~

    :copyright: 2010 by Bjoern Schulz <bjoern.schulz.92@gmail.com>
    :license: MIT, see LICENSE for more details
'''

from sys import stdout
from itertools import izip
import operator
import mania
from mania.types import (
    FunctionWrapper, Type, Integer, Real, String, Name, List, Struct,
    Boolean, true, false, nil
)

class Builtins(dict):
    
    def __setitem__(self, name, value):
        dict.__setitem__(self, name, FunctionWrapper(value))
    
    def __call__(self, name):
        def wrapper(func):
            self[name] = func
        return wrapper

builtins = Builtins({
    'apply': FunctionWrapper(lambda f, *a: f.call(*a)),
    'input': FunctionWrapper(lambda: String(raw_input())),
    'length': FunctionWrapper(lambda s: s.length()),
    'first': FunctionWrapper(lambda s: s.first()),
    'rest': FunctionWrapper(lambda s: s.rest()),
    'push': FunctionWrapper(lambda s, i: s.push(i)),
    'pop': FunctionWrapper(lambda s: s.pop()),
    'index': FunctionWrapper(lambda s, i: s.index(i)),
    'concat': FunctionWrapper(lambda s, o: s.concat(o)),
    'join': FunctionWrapper(lambda s, *a: s.join(a)),
    'format': FunctionWrapper(lambda s, *a: s.format(a)),
    'inc': FunctionWrapper(lambda o: o.add(Integer(1))),
    'dec': FunctionWrapper(lambda o: o.sub(Integer(1))),
    'dir': FunctionWrapper(lambda m: List([Name(k) for k in m.value])),
    '*': FunctionWrapper(lambda a, b: a.mul(b)),
    '//': FunctionWrapper(lambda a, b: a.div(b)),
    '/': FunctionWrapper(lambda a, b: a.truediv(b)),
    '%': FunctionWrapper(lambda a, b: a.mod(b)),
    '**': FunctionWrapper(lambda a, b: a.pow(b)),
    '|': FunctionWrapper(lambda a, b: a.bitwise_or(b)),
    '^': FunctionWrapper(lambda a, b: a.bitwise_xor(b)),
    '&': FunctionWrapper(lambda a, b: a.bitwise_and(b)),
    '~': FunctionWrapper(lambda a: a.bitwise_not()),
    '<<': FunctionWrapper(lambda a, b: a.lshift(b)),
    '>>': FunctionWrapper(lambda a, b: a.rshift(b)),
    '=': FunctionWrapper(lambda a, b: a.equals(b)),
    '!=': FunctionWrapper(lambda a, b: a.not_equals(b)),
    'not': FunctionWrapper(lambda a: false if a.to_boolean() else true),
    '>': FunctionWrapper(lambda a, b: a.greater(b)),
    '>=': FunctionWrapper(lambda a, b: a.greater_equals(b)),
    '<': FunctionWrapper(lambda a, b: a.lower(b)),
    '<=': FunctionWrapper(lambda a, b: a.lower_equals(b)),
    'is': FunctionWrapper(lambda a, b: a.equivalent(b)),
    'in': FunctionWrapper(lambda a, b: a.contains(b)),
    'typeof?': FunctionWrapper(lambda s: type(s)),
    'istype?': FunctionWrapper(lambda s, o: true if type(s) is o else false),
    'integer': Integer,
    'real': Real,
    'string': String,
    'name': Name,
    'boolean': Boolean,
    'list': List,
    'struct': Struct
})

@builtins('puts')
def puts(*args):
    for s in args:
        print s.to_string().value
    return nil

@builtins('print')
def print_(s):
    stdout.write(s.to_string().value)
    return nil

@builtins('+')
def add(a, b=None):
    if b is None:
        return a.pos()
    return a.add(b)

@builtins('-')
def sub(a, b=None):
    if b is None:
        return a.neg()
    return a.sub(b)

@builtins('map')
def map_(func, *iterables):
    result = List()
    for args in izip(*iterables):
        result.push(func.call(*args))
    return result

@builtins('sum')
def sum_(values, start=0):
    for value in values:
        start += value
    return start

@builtins('all')
def all_(values):
    for item in values:
        if not item.to_boolean():
            return false
    return true

@builtins('any')
def any_(values):
    for item in values:
        if item.to_boolean():
            return true
    return false

@builtins('to')
def to(type, obj):
    types = {
        Integer: lambda s: s.to_integer(),
        Real: lambda s: s.to_real(),
        String: lambda s: s.to_string(),
        Name: lambda s: s.to_name(),
        Boolean: lambda s: s.to_boolean(),
        List: lambda s: s.to_list(),
        Struct: lambda s: s.to_struct()
    }
    return types[type](obj)