import collections
import copy
from math import ceil

from mapmyfitness.exceptions import PageNotAnInteger, EmptyPage


class Paginator(object):
    def __init__(self, initial_object_list, per_page, total_count,
                 searchable, original_kwargs, orphans=0):
        self.initial_object_list = initial_object_list
        self.per_page = per_page
        self.total_count = total_count
        self.searchable = searchable
        self.original_kwargs = original_kwargs
        self.orphans = int(orphans)
        self._num_pages = None

    def validate_number(self, number):
        try:
            number = int(number)
        except (TypeError, ValueError):
            raise PageNotAnInteger('That page number is not an integer')
        if number < 1:
            raise EmptyPage('That page number is less than 1')
        if number > self.num_pages:
            raise EmptyPage('That page contains no results')
        return number

    def page(self, number):
        number = self.validate_number(number)
        bottom = (number - 1) * self.per_page
        top = bottom + self.per_page
        if top + self.orphans >= self.count:
            top = self.count
        if number == 1:
            objects = self.initial_object_list
        else:
            params = copy.deepcopy(self.original_kwargs)
            params.update({'offset': bottom, 'limit': self.per_page})
            objects = self.searchable._search(params)
        return self._get_page(objects, number, self)

    def _get_page(self, *args, **kwargs):
            return Page(*args, **kwargs)

    def _get_count(self):
        return self.total_count
    count = property(_get_count)

    def _get_num_pages(self):
        if self._num_pages is None:
            hits = max(1, self.count - self.orphans)
            self._num_pages = int(ceil(hits / float(self.per_page)))
        return self._num_pages
    num_pages = property(_get_num_pages)

    def _get_page_range(self):
        return list(range(1, self.num_pages + 1))
    page_range = property(_get_page_range)


class Page(collections.Sequence):

    def __init__(self, object_list, number, paginator):
        self.object_list = object_list
        self.number = number
        self.paginator = paginator

    def __repr__(self):
        return '<Page %s of %s>' % (self.number, self.paginator.num_pages)

    def __len__(self):
        return len(self.object_list)

    def __getitem__(self, index):
        return self.object_list[index]

    def has_next(self):
        return self.number < self.paginator.num_pages

    def has_previous(self):
        return self.number > 1

    def has_other_pages(self):
        return self.has_previous() or self.has_next()

    def next_page_number(self):
        return self.paginator.validate_number(self.number + 1)

    def previous_page_number(self):
        return self.paginator.validate_number(self.number - 1)

    def start_index(self):
        # Special case, return zero if no items.
        if self.paginator.count == 0:
            return 0
        return (self.paginator.per_page * (self.number - 1)) + 1

    def end_index(self):
        # Special case for the last page because there can be orphans.
        if self.number == self.paginator.num_pages:
            return self.paginator.count
        return self.number * self.paginator.per_page
