import os
import jinja2
import inspect
from run import Module
from run.plugins.io import DialogModule  # @UnresolvedImport
from run.plugins.render import RenderFileTask, RenderStringTask  # @UnresolvedImport


class ProcessModule(Module):
    """Process mario project.
    """

    # Public

    def __init__(self):
        self.__update_pythonpath()

    # Modules

    dialog_context = DialogModule()

    # Tasks

    def process_files(self):
        root = 'meta'
        if os.path.isdir(root):
            for dirpath, _, filenames in os.walk(root):
                for filename in sorted(filenames):
                    if filename.startswith('_'):
                        continue
                    source = os.path.join(dirpath, filename)
                    target = os.path.relpath(source, start=root)
                    content = self.render_file(filepath=source)
                    self.write_file(target, content)

    def process_filenames(self):
        root = '.'
        for dirpath, dirnames, filenames in os.walk(root, topdown=False):
            for name in dirnames + filenames:
                new_name = self.render_string(name)
                if name != new_name:
                    path = os.path.join(dirpath, name)
                    new_path = os.path.join(dirpath, new_name)
                    # TODO: pass exceptions on existent files
                    os.rename(path, new_path)

    def process_runfile(self):
        path = 'runfile.py'
        content = self.render_file(path)
        self.write_file(path, content)

    process_file = RenderFileTask(
        trim_blocks=True,
        lstrip_blocks=True,
        loader=jinja2.ChoiceLoader([
            jinja2.FileSystemLoader('.'),
            jinja2.PrefixLoader({
                'mario': jinja2.PackageLoader(
                    'mario', 'templates')})]))

    process_string = RenderStringTask()

    def write_file(self, path, content):
        dirname = os.path.dirname(path)
        if dirname:
            os.makedirs(dirname, exist_ok=True)
        with open(path, 'w') as file:
            file.write(content)

    # Private

    def __update_pythonpath(self):
        os.environ['PYTHONPATH'] = ':'.join([
            os.environ.get('PYTHONPATH', ''),
            os.path.abspath(os.path.dirname(inspect.getfile(type(self))))])
