import os
import logging.config
from clyde import Command, Option, ManpageFormatter, mixin
from .project import Project
from .settings import settings
version = '0.25.0'  # TODO: syncronize


class Program(Command):
    """Mario main program.
    """

    # Public

    meta_formatter = ManpageFormatter

    # Mixins

    @mixin
    def initiate_logging(self):
        logging.config.dictConfig(settings.logging_config)
        logger = logging.getLogger()
        if self.verbose:
            logger.setLevel(logging.DEBUG)
        if self.quiet:
            logger.setLevel(logging.ERROR)

    @mixin(require='help')
    def print_help(self):
        print(self.meta_format('help'))
        exit()

    @mixin(require='version')
    def print_version(self):
        print(version)
        exit()

    @mixin(require=Exception)
    def process_exception(self, exception):
        logger = logging.getLogger(__name__)
        logger.error(str(exception), exc_info=self.verbose)
        exit(1)

    # Options

    basedir = Option(
        flags=['-b', '--basedir'],
        default=os.getcwd(),
        help='Base directory to work.',
    )

    help = Option(
        action='store_true',
        flags=['-h', '--help'],
        help='Display this help message.',
    )

    quiet = Option(
        action='store_true',
        flags=['-q', '--quiet'],
        help='Enable quiet mode.',
    )

    verbose = Option(
        action='store_true',
        flags=['-v', '--verbose'],
        help='Enable verbose mode.',
    )

    version = Option(
        action='store_true',
        flags=['-V', '--version'],
        help='Display the program version.',
    )

    # Subcommands

    class bind(Command):
        """Bind project origin to remote.
        """

        # Public

        def meta_execute(self, remote):
            project = Project(basedir=self.basedir)
            project.bind(remote)

    class init(Command):
        """Init project.
        """

        # Public

        def meta_execute(self, origin=None):
            project = Project(basedir=self.basedir, create=True)
            if origin:
                project.bind(origin)
                project.sync()
                project.make(command=self.command)

        # Options

        command = Option(
            flags=['-c', '--command'],
            default=settings.command,
            help='Shell command to bootstrap project.',
        )

    class make(Command):
        """Make project.
        """

        # Public

        def meta_execute(self):
            project = Project(basedir=self.basedir)
            project.make(command=self.command)

        # Options

        command = Option(
            flags=['-c', '--command'],
            default=settings.command,
            help='Shell command to bootstrap project.',
        )

    class sync(Command):
        """Sync project's origin with remote.
        """

        # Public

        def meta_execute(self):
            project = Project(basedir=self.basedir)
            project.sync()


program = Program(name='mario')
