import os
import shutil
from subprocess import check_call
from tempfile import TemporaryDirectory


class Origin:
    """Origin wrapper.
    """

    # Public

    def __init__(self, sysdir):
        self.__path = os.path.join(sysdir, self.__ORIGIN_DIR)
        if not self.__check():
            self.__create()

    def copy(self, target):
        source = os.path.join(self.__path, self.__SOURCES_DIR)
        for item in os.listdir(source):
            item_source = os.path.join(source, item)
            item_target = os.path.join(target, item)
            try:
                if os.path.isdir(item_source):
                    shutil.copytree(item_source, item_target)
                else:
                    shutil.copy2(item_source, item_target)
            except FileExistsError:
                pass

    def sync(self, remote):
        try:
            shutil.rmtree(self.__path)
            with TemporaryDirectory(prefix='') as dirpath:
                check_call(['git', 'clone', remote, dirpath])
                source = os.path.join(dirpath, self.__ORIGIN_DIR)
                shutil.copytree(source, self.__path)
        except Exception as exception:
            raise RuntimeError(
                'Can\'t sync origin because of {exception}'.
                format(exception=exception))

    # Private

    __ORIGIN_DIR = 'origin'
    __SOURCES_DIR = 'sources'

    def __check(self):
        return os.path.exists(self.__path)

    def __create(self):
        os.makedirs(self.__path, exist_ok=True)
