import os
import logging
import subprocess
from sugarbowl import cachedproperty
from ..settings import settings
from .config import Config
from .content import Content
from .origin import Origin


class Project:
    """Project wrapper.
    """

    # Public

    def __init__(self, *, basedir=None, create=False):
        if basedir is None:
            basedir = os.getcwd()
        self.__basedir = basedir
        self.__dirpath = os.path.join(self.__basedir, settings.metadir)
        if not self.__check():
            if not create:
                raise RuntimeError('Not a mario project (use mario init)')
            self.__create()
            logger.info('Initiated mario project')
        elif create:
            logger.info('Reinitiated mario project')

    def bind(self, remote):
        self.__config.set('remote', remote)
        logger.info('Bound origin to "%s"', remote)

    def make(self, *, command=None):
        self.__origin.copy(self.__basedir)
        if command:
            subprocess.check_call(
                command, shell=True, cwd=self.__basedir)
        logger.info('Made project from origin')

    def sync(self):
        remote = self.__config.get('remote')
        if remote:
            self.__origin.sync(remote)
            logger.info('Synced with "%s"', remote)
        else:
            logger.info('No remote to sync with (use mario bind)')

    def process(self, render):
        self.__content.process(render)

    # Private

    def __check(self):
        return os.path.exists(self.__dirpath)

    def __create(self):
        os.makedirs(self.__dirpath, exist_ok=True)

    @cachedproperty
    def __config(self):
        return Config(basedir=self.__basedir)

    @cachedproperty
    def __content(self):
        return Content(basedir=self.__basedir)

    @cachedproperty
    def __origin(self):
        return Origin(basedir=self.__basedir)


logger = logging.getLogger(__name__)
