import sys
from nose.tools import *
from StringIO import StringIO
import sqlparse
from sqlparse import tokens as T
from sqlparse.sql import Token, TokenList, Parenthesis, Identifier
from mbdata.utils.sql import (
    group_parentheses,
    parse_statements,
    Set,
    CreateTable,
    CreateType,
)


def test_group_parentheses():
    tokens = [
        Token(T.Keyword, 'CREATE'),
        Token(T.Whitespace, ' '),
        Token(T.Keyword, 'TABLE'),
        Token(T.Whitespace, ' '),
        Token(T.Name, 'table_name'),
        Token(T.Whitespace, ' '),
        Token(T.Punctuation, '('),
        Token(T.Name, 'id'),
        Token(T.Whitespace, ' '),
        Token(T.Keyword, 'SERIAL'),
        Token(T.Whitespace, ' '),
        Token(T.Keyword, 'CHECK'),
        Token(T.Punctuation, '('),
        Token(T.Name, 'id'),
        Token(T.Operator, '='),
        Token(T.Number, '0'),
        Token(T.Punctuation, ')'),
        Token(T.Punctuation, ')'),
        Token(T.Punctuation, ';'),
    ]

    expected_tokens = TokenList([
        Token(T.Keyword, 'CREATE'),
        Token(T.Keyword, 'TABLE'),
        Token(T.Name, 'table_name'),
        Parenthesis([
            Token(T.Punctuation, '('),
            Token(T.Name, 'id'),
            Token(T.Keyword, 'SERIAL'),
            Token(T.Keyword, 'CHECK'),
            Parenthesis([
                Token(T.Punctuation, '('),
                Token(T.Name, 'id'),
                Token(T.Operator, '='),
                Token(T.Number, '0'),
                Token(T.Punctuation, ')'),
            ]),
            Token(T.Punctuation, ')'),
        ]),
        Token(T.Punctuation, ';'),
    ])

    grouped_tokens = group_parentheses(tokens)

    stdout = sys.stdout
    try:
        sys.stdout = StringIO()
        expected_tokens._pprint_tree()
        a = sys.stdout.getvalue()
        sys.stdout = StringIO()
        grouped_tokens._pprint_tree()
        b = sys.stdout.getvalue()
    finally:
        sys.stdout = stdout

    assert_multi_line_equal(a, b)


def test_parse_statements():
    sql = '''
SET search_path = 'cover_art_archive';

CREATE TABLE table_name (
    id SERIAL, -- PK
    name VARCHAR
);

CREATE TYPE FLUENCY AS ENUM ('basic', 'intermediate', 'advanced', 'native');
    '''
    statements = sqlparse.parse(sql)
    for statement in statements:
        statement._pprint_tree()
        print
    statements = parse_statements(statements)
    for statement in statements:
        print repr(statement)


def test_set_statement():
    sql = "SET search_path = 'cover_art_archive';"
    statement = parse_statements(sqlparse.parse(sql)).next()

    assert_is_instance(statement, Set)
    assert_equals('search_path', statement.get_name())
    assert_equals('cover_art_archive', statement.get_value())


def test_set_statement_without_quotes():
    sql = "SET search_path = cover_art_archive;"
    statement = parse_statements(sqlparse.parse(sql)).next()

    assert_is_instance(statement, Set)
    assert_equals('search_path', statement.get_name())
    assert_equals('cover_art_archive', statement.get_value())


def test_set_statement_with_to():
    sql = "SET search_path TO 'cover_art_archive';"
    statement = parse_statements(sqlparse.parse(sql)).next()

    assert_is_instance(statement, Set)
    assert_equals('search_path', statement.get_name())
    assert_equals('cover_art_archive', statement.get_value())


def test_create_type_statement():
    sql = "CREATE TYPE FLUENCY AS ENUM ('basic', 'intermediate');"
    statement = parse_statements(sqlparse.parse(sql)).next()

    assert_is_instance(statement, CreateType)
    assert_equals('FLUENCY', statement.get_name())
    assert_equals(['basic', 'intermediate'], statement.get_enum_labels())


def test_create_table_statement():
    sql = '''
CREATE TABLE table_name (
    id SERIAL, -- PK
    name VARCHAR(100) NOT NULL,
    created TIMESTAMP WITH TIME ZONE DEFAULT now() NOT NULL
);
    '''
    statement = parse_statements(sqlparse.parse(sql)).next()

    assert_is_instance(statement, CreateTable)
    assert_equals('table_name', statement.get_name())

    columns = list(statement.get_columns())
    assert_equals(3, len(columns))

    column = columns[0]
    assert_equals('id', column.get_name())
    assert_equals('SERIAL', column.get_type())
    assert_equals(None, column.get_default_value())
    assert_equals(['-- PK'], column.get_comments())
    assert_equals(False, column.is_not_null())
    assert_equals(None, column.get_check_constraint())

    column = columns[1]
    assert_equals('name', column.get_name())
    assert_equals('VARCHAR(100)', column.get_type())
    assert_equals(None, column.get_default_value())
    assert_equals([], column.get_comments())
    assert_equals(True, column.is_not_null())
    assert_equals(None, column.get_check_constraint())

    column = columns[2]
    assert_equals('created', column.get_name())
    assert_equals('TIMESTAMP WITH TIME ZONE', column.get_type())
    assert_equals('now()', column.get_default_value())
    assert_equals([], column.get_comments())
    assert_equals(True, column.is_not_null())
    assert_equals(None, column.get_check_constraint())


def test_create_table_statement_check_constraint():
    sql = '''CREATE TABLE table_name (column INTEGER(2) NOT NULL DEFAULT 0 CHECK (edits_pending > 0)); '''
    statement = parse_statements(sqlparse.parse(sql)).next()

    assert_is_instance(statement, CreateTable)
    columns = list(statement.get_columns())
    assert_equal(1, len(columns))

    column = columns[0]
    check = column.get_check_constraint()
    assert_true(check)
    assert_equal(None, check.get_name())
    assert_equal('edits_pending>0', str(check.get_body()))


def test_create_table_statement_named_check_constraint():
    sql = '''CREATE TABLE table_name (column INTEGER(2) NOT NULL DEFAULT 0 CONSTRAINT check_column CHECK (edits_pending > 0)); '''
    statement = parse_statements(sqlparse.parse(sql)).next()

    assert_is_instance(statement, CreateTable)
    columns = list(statement.get_columns())
    assert_equal(1, len(columns))

    column = columns[0]
    check = column.get_check_constraint()
    assert_true(check)
    assert_equal('check_column', check.get_name())
    assert_equal('edits_pending>0', str(check.get_body()))

