"""
Copyright (c) 2012, Hsiaoming Yang <http://lepture.com>

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above
     copyright notice, this list of conditions and the following
     disclaimer in the documentation and/or other materials provided
     with the distribution.
   * Neither the name of the author nor the names of its contributors
     may be used to endorse or promote products derived from this
     software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
"""
import os
import logging
import time
import mimetypes
import webbrowser
import hashlib
import pkg_resources
from tornado import ioloop
from tornado import escape
from tornado import websocket
from tornado.web import RequestHandler, Application
from tornado.util import ObjectDict

from task import Task

LIVERELOAD = pkg_resources.resource_filename(__name__, 'livereload.js')

class LiveReloadHandler(websocket.WebSocketHandler):
    waiters = set()
    _last_reload_time = None
    
    def __init__(self, *args, **kwargs):
            self.file = kwargs.pop('file')
            self.func = kwargs.pop('func')
            super(LiveReloadHandler, self).__init__(*args, **kwargs)

    def allow_draft76(self):
        return True

    def on_close(self):
        if self in LiveReloadHandler.waiters:
            LiveReloadHandler.waiters.remove(self)

    def send_message(self, message):
        if isinstance(message, dict):
            message = escape.json_encode(message)

        try:
            self.write_message(message)
        except:
            logging.error('Error sending message', exc_info=True)

    def poll_tasks(self):
        changes = Task.watch()
        if not changes:
            return
        self.watch_tasks()

    def watch_tasks(self):
        if time.time() - self._last_reload_time < 3:
            # if you changed lot of files in one time
            # it will refresh too many times
            logging.info('ignore this reload action')
            return

        logging.info('Reload %s waiters', len(self.waiters))

        msg = {
            'command': 'reload',
            'path': Task.last_modified or '*',
            'liveCSS': True
        }

        self._last_reload_time = time.time()
        for waiter in LiveReloadHandler.waiters:
            try:
                waiter.write_message(msg)
            except:
                logging.error('Error sending message', exc_info=True)
                LiveReloadHandler.waiters.remove(waiter)

    def on_message(self, message):
        """Handshake with livereload.js

        1. client send 'hello'
        2. server reply 'hello'
        3. client send 'info'

        http://help.livereload.com/kb/ecosystem/livereload-protocol
        """
        message = ObjectDict(escape.json_decode(message))
        if message.command == 'hello':
            handshake = {}
            handshake['command'] = 'hello'
            protocols = message.protocols
            protocols.append(
                'http://livereload.com/protocols/2.x-remote-control'
            )
            handshake['protocols'] = protocols
            handshake['serverName'] = 'livereload-tornado'
            self.send_message(handshake)

        if message.command == 'info' and 'url' in message:
            logging.info('Browser Connected: %s' % message.url)
            LiveReloadHandler.waiters.add(self)
            if not LiveReloadHandler._last_reload_time:
                # Task.add(os.getcwd())
                Task.add(self.file, self.func)
                Task.add('extras')

                LiveReloadHandler._last_reload_time = time.time()
                logging.info('Start watching changes')
                if not Task.start(self.watch_tasks):
                    ioloop.PeriodicCallback(self.poll_tasks, 800).start()

class IndexHandler(RequestHandler):
    
    def inject_livereload(self):
        if self.mime_type != 'text/html':
            return
        ua = self.request.headers.get('User-Agent', 'bot').lower()
        if 'msie' not in ua:
            self.write('<script src="/livereload.js"></script>')
    
    def get(self, path='/'):
        abspath = os.path.join(os.getcwd(), path.lstrip('/'))
        mime_type, encoding = mimetypes.guess_type(abspath)
        if not mime_type:
            mime_type = 'text/html'

        self.mime_type = mime_type
        self.set_header('Content-Type', mime_type)
        
        if not os.path.exists(abspath):
            filepath = abspath + '.html'
        
        if not os.path.exists(abspath):
            self.send_error(404)
            return
            
        if self.mime_type == 'text/html':
            f = open(abspath)
            data = f.read()
            f.close()
            before, after = data.split('</head>')
            self.write(before)
            self.inject_livereload()
            self.write('</head>')
            self.write(after)
        else:
            f = open(abspath, 'rb')
            data = f.read()
            f.close()
            self.write(data)
        
        hasher = hashlib.sha1()
        hasher.update(data)
        self.set_header('Etag', '"%s"' % hasher.hexdigest())
        return    

class LiveReloadJSHandler(RequestHandler):
    def get(self):
        f = open(LIVERELOAD)
        self.set_header('Content-Type', 'application/javascript')
        self.write(f.read())
        f.close()

def start_livereload(file_to_serve, file_to_watch, func=None):
    handlers = [
        (r'/livereload', LiveReloadHandler, {'file': file_to_watch, 'func': func}),
        (r'/livereload.js', LiveReloadJSHandler),
        (r'(.*)', IndexHandler),
    ]
    app = Application(handlers=handlers)
    app.listen(35729)
    logging.info('Serving file {0} on 127.0.0.1:35729/{0}'.format(file_to_serve))
    webbrowser.open('http://127.0.0.1:35729/{0}'.format(file_to_serve), new=2, autoraise=True)
    try:
        ioloop.IOLoop.instance().start()
    except KeyboardInterrupt:
        print('Shutting down...')